<?php
namespace Mewz\WCAS\Actions\Front;

use Mewz\Framework\Base\Action;
use Mewz\WCAS\Util\Attributes;
use Mewz\WCAS\Util\Cart;
use Mewz\WCAS\Util\Products;

class CartAdd extends Action
{
	protected $cart_contents;

	public function __hooks()
	{
		add_filter('woocommerce_add_cart_item', [$this, 'add_cart_item'], 5, 2);

		// validate stock limits *after* all other `add_to_cart` validations
		add_action('woocommerce_add_to_cart', [$this, 'add_to_cart'], -100, 6);
		add_action('woocommerce_bundled_add_to_cart', [$this, 'add_to_cart'], -100, 6);
	}

	public function add_cart_item($item, $key)
	{
		Cart::set_item_product_variation_data($item);

		return $item;
	}

	public function add_to_cart($cart_item_key, $product_id, $quantity = 1, $variation_id = 0, $variation = [], $cart_item_data = [])
	{
		$product_id = (int)$product_id;
		$variation_id = (int)$variation_id;

		$product = wc_get_product($variation_id ?: $product_id);
		if (!$product) return;

		if ($variation && $product instanceof \WC_Product_Variation && Attributes::has_catchall($product->get_attributes())) {
			$product->mewz_wcas_variation = $variation;
		}

		if (Products::is_product_excluded($product)) return;

		do_action('mewz_wcas_before_validate_add_to_cart', $product, $cart_item_key, $quantity, $variation);

		$valid = true;

		if ($product->managing_stock() && !$product->backorders_allowed()) {
			$this->revert_cart_contents($cart_item_key, $quantity);

			$valid = $this->validate_adding_product($product, $quantity, $cart_item_key);
			$valid = apply_filters('mewz_wcas_validate_add_to_cart', $valid, $product, $cart_item_key, $quantity, $variation);

			$this->restore_cart_contents();
		}

		do_action('mewz_wcas_after_validate_add_to_cart', $valid, $product, $cart_item_key, $quantity, $variation);

		if ($valid instanceof \WP_Error) {
			$this->undo_added_item($cart_item_key, $quantity);
			throw new \Exception($valid->get_error_message());
		}
	}

	public function validate_adding_product(\WC_Product $product, $quantity, $cart_item_key)
	{
		if ($product instanceof \WC_Product_Variation) {
			if (property_exists($product, 'mewz_wcas_variation')) {
				// check stock quantities again for catch-all variations now that the variation attributes are available to do a proper stock-limit check
				$result = Cart::validate_adding_stock($product, $quantity);
				if ($result !== true) return $result;
			}

			// check that there is enough stock for variations using product-level stock
			$result = Cart::validate_adding_parent_stock($product, $quantity);
			if ($result !== true) return $result;
		}

		// check that enough attribute stock is available for this item and other items in the cart (shared stock quantity)
		$result = Cart::validate_adding_shared_attribute_stock($product, $quantity, $cart_item_key);
		if ($result !== true) return $result;

		return true;
	}

	protected function revert_cart_contents($cart_item_key, $quantity)
	{
		if ($this->cart_contents !== null) return;

		$cart = WC()->cart;

		if (!isset($cart->cart_contents[$cart_item_key])) return;

		$this->cart_contents = $cart->cart_contents;
		$cart->cart_contents[$cart_item_key]['quantity'] -= $quantity;

		if ($cart->cart_contents[$cart_item_key]['quantity'] <= 0) {
			unset($cart->cart_contents[$cart_item_key]);
		}
	}

	protected function restore_cart_contents()
	{
		if ($this->cart_contents === null) return;

		WC()->cart->cart_contents = $this->cart_contents;
		$this->cart_contents = null;
	}

	protected function undo_added_item($cart_item_key, $quantity)
	{
		$cart = WC()->cart;

		$cart_items = $this->get_all_related_cart_items($cart->cart_contents, $cart_item_key);
		if (!$cart_items) return;

		foreach ($cart_items as $item_key => $item) {
			$cart->set_quantity($item_key, $item['quantity'] - $quantity);
		}

		$cart->cart_contents = apply_filters('woocommerce_cart_contents_changed', $cart->cart_contents);
	}

	protected function get_all_related_cart_items($cart_contents, $cart_item_key)
	{
		if (empty($cart_contents[$cart_item_key])) {
			return [];
		}

		$items[$cart_item_key] = $cart_contents[$cart_item_key];

		if (!empty($cart_item['bundled_by']) && !empty($cart_contents[$cart_item['bundled_by']])) {
			$items[] = $bundle = $cart_contents[$cart_item['bundled_by']];

			if (!empty($bundle['bundled_items'])) {
				foreach ($bundle['bundled_items'] as $bundled_item_key) {
				    if (!empty($cart_contents[$bundled_item_key])) {
					    $items[] = $cart_contents[$bundled_item_key];
				    }
				}
			}
		}

		return $items;
	}
}
