<?php
namespace Mewz\WCAS\Actions\Admin\Stock;

use Mewz\Framework\Base\Action;
use Mewz\Framework\Compatibility\Multilang;
use Mewz\Framework\Util\Admin;
use Mewz\WCAS\Models\AttributeStock;
use Mewz\WCAS\Util\Attributes;
use Mewz\WCAS\Util\Matches;
use Mewz\WCAS\Util\Products;

class StockEdit extends Action
{
	public function __run()
	{
		$this->spoof_backlink_referer();
	}

	public function __hooks()
	{
		add_filter('woocommerce_screen_ids', [$this, 'woocommerce_screen_ids']);

		add_action('admin_enqueue_scripts', [$this, 'admin_enqueue_scripts']);
		add_action('add_meta_boxes', [$this, 'add_meta_boxes']);

		add_filter('post_updated_messages', [$this, 'post_updated_messages']);

		add_filter('admin_body_class', [$this, 'admin_body_class'], 100);
	}

	public function __scripts()
	{
		$this->scripts->enqueue_bundle('@admin/stock-edit');
		$this->scripts->enqueue_js('wc-admin-meta-boxes');
	}

	public function woocommerce_screen_ids($screen_ids)
	{
		$screen_ids[] = 'mewz_attribute_stock';

		return $screen_ids;
	}

	public function admin_enqueue_scripts()
	{
		wp_dequeue_script('autosave');
	}

	public function add_meta_boxes()
	{
		add_meta_box('mewz-wcas-status-metabox', __('Status'), [$this, 'display_status_metabox'], 'mewz_attribute_stock', 'side', 'high');
		add_meta_box('mewz-wcas-details-metabox', __('Stock details', 'woocommerce-attribute-stock'), [$this, 'display_details_metabox'], 'mewz_attribute_stock', 'normal', 'high');
		add_meta_box('mewz-wcas-matches-metabox', __('Match rules', 'woocommerce-attribute-stock'), [$this, 'display_matches_metabox'], 'mewz_attribute_stock', 'normal');

		remove_meta_box('submitdiv', 'mewz_attribute_stock', 'side');
		remove_meta_box('slugdiv', 'mewz_attribute_stock', 'normal');
	}

	public function display_status_metabox($post)
	{
		$stock = AttributeStock::instance($post, 'edit');

		$this->view->render('admin/stock/metabox-status', ['stock' => $stock]);
	}

	public function display_details_metabox($post)
	{
		$stock = AttributeStock::instance($post, 'edit');

		$tabs = [
			'inventory' => __('Inventory', 'woocommerce-attribute-stock'),
			'settings' => __('Settings', 'woocommerce-attribute-stock'),
		];

		foreach ($tabs as $key => $tab) {
		    add_action('mewz_wcas_stock_details_panel_' . $key, [$this, 'details_panel_' . $key], 10, 2);
		}

		$tabs = apply_filters('mewz_wcas_stock_details_tabs', $tabs);

		foreach ($tabs as $key => $tab) {
			if (is_string($tab)) {
				$tabs[$key] = ['label' => $tab];
			}
		}

		$this->view->render('admin/stock/metabox-details', [
			'stock' => $stock,
			'tabs' => $tabs,
		]);
	}

	public function display_matches_metabox($post)
	{
		$stock = AttributeStock::instance($post, 'edit');

		$tabs = [
			'attributes' => __('Attributes', 'woocommerce-attribute-stock'),
			'filters' => __('Filters', 'woocommerce-attribute-stock'),
		];

		foreach ($tabs as $key => $tab) {
			add_action('mewz_wcas_matches_panel_' . $key, [$this, 'matches_panel_' . $key], 10, 2);
		}

		$tabs = apply_filters('mewz_wcas_matches_tabs', $tabs);

		foreach ($tabs as $key => $tab) {
			if (is_string($tab)) {
				$tabs[$key] = ['label' => $tab];
			}
		}

		if (Multilang::active()) {
			$reset_lang = Multilang::set_lang(Multilang::get_lang('default'));
		}

		$this->view->render('admin/stock/metabox-matches', [
			'stock' => $stock,
			'tabs' => $tabs,
		]);

		if (!empty($reset_lang)) {
			Multilang::reset_lang();
		}
	}

	public function details_panel_inventory(AttributeStock $stock, $tab)
	{
		$this->view->render('admin/stock/panel-inventory', ['stock' => $stock]);
	}

	public function details_panel_settings(AttributeStock $stock, $tab)
	{
		$this->view->render('admin/stock/panel-settings', ['stock' => $stock]);
	}

	public function matches_panel_attributes(AttributeStock $stock, $tab)
	{
		$this->view->render('admin/stock/panel-attributes');

		$this->scripts->export_js('@admin/stock-edit', 'mewz_wcas_data', [
			'attributes' => $this->get_attribute_data(),
			'match_sets' => array_values(Matches::get_sets($stock->id(), 'edit')),
			'locale' => str_replace('_', '-', get_bloginfo('language')),
		]);
	}

	public function matches_panel_filters(AttributeStock $stock, $tab)
	{
		$this->view->render('admin/stock/panel-filters', [
			'stock' => $stock,
			'products' => $this->get_product_options($stock->products()),
			'exclude_products' => $this->get_product_options($stock->exclude_products()),
			'categories' => Admin::get_category_options(),
			'product_types' => Products::get_product_types(),
		]);
	}

	public function get_attribute_data()
	{
		$attributes = Attributes::get_attributes();

		foreach ($attributes as &$attr) {
			$attr['label'] = html_entity_decode($attr['label']);
			$attr['taxonomy'] = 'pa_' . $attr['name'];

			$terms = get_terms([
				'taxonomy' => $attr['taxonomy'],
				'fields' => 'id=>name',
				'hide_empty' => false,
				'update_term_meta_cache' => false,
			]);

			foreach ($terms as $term_id => $term_label) {
				$term_label = trim($term_label);

				if ($term_label === '') {
					$term_label = Attributes::get_term_prop($term_id, $attr['taxonomy'], 'slug') ?: (string)$term_id;
				} else {
					$term_label = html_entity_decode($term_label);
				}

				$attr['terms'][] = [$term_id, $term_label];
			}

			unset($attr['id'], $attr['name']);
		}

		return $attributes;
	}

	public function get_product_options($product_ids)
	{
		$options = [];

		foreach ($product_ids as $product_id) {
			if ($product = wc_get_product($product_id)) {
				$options[$product_id] = htmlspecialchars(Products::get_formatted_product_name($product));
			}
		}

		return $options;
	}

	public function post_updated_messages($messages)
	{
		global $post;

		$stock = AttributeStock::instance($post);

		$messages['mewz_attribute_stock'] = [
			0 => '', // Unused. Messages start at index 1.
			1 => __('Attribute stock updated.', 'woocommerce-attribute-stock'),
			2 => __('Attribute stock field updated.', 'woocommerce-attribute-stock'),
			3 => __('Attribute stock field deleted.', 'woocommerce-attribute-stock'),
			4 => __('Attribute stock updated.', 'woocommerce-attribute-stock'),
			5 => __('Revision restored.', 'woocommerce-attribute-stock'),
			6 => __('Attribute stock updated.', 'woocommerce-attribute-stock'),
			7 => __('Attribute stock saved.', 'woocommerce-attribute-stock'),
			8 => __('Attribute stock submitted.', 'woocommerce-attribute-stock'),
			9 => sprintf(
				/* translators: %s: date */
				__( 'Attribute stock scheduled for: %s.', 'woocommerce-attribute-stock' ),
				'<strong>' . $stock->created(false, 'admin-full') . '</strong>'
			),
			10 => __('Attribute stock updated.', 'woocommerce-attribute-stock'),
		];

		if ($back_link = $this->get_back_link()) {
			$messages['mewz_attribute_stock'][4] .= '</p><p>' . $back_link;
		}

		return $messages;
	}

	public function get_back_link()
	{
		$referer = wp_get_referer();

		if ($referer) {
			if (strpos($referer, '/edit.php') !== false && strpos($referer, 'post_type=mewz_attribute_stock') !== false) {
				$text = __('Back to attribute stock', 'woocommerce-attribute-stock');
			}
			elseif (strpos($referer, '/term.php?taxonomy=') !== false) {
				if (preg_match('/tag_ID=(\d+)/', $referer, $matches) && isset($matches[1]) && $term = get_term($matches[1])) {
					$text = sprintf(__('Back to "%s" attribute term', 'woocommerce-attribute-stock'), $term->name);
				} else {
					$text = __('Back to attribute term', 'woocommerce-attribute-stock');
				}
			}
			elseif (strpos($referer, '/edit-tags.php?taxonomy=') !== false) {
				if (preg_match('/taxonomy=(\w+)/', $referer, $matches) && isset($matches[1]) && $attribute = Attributes::get_attribute($matches[1])) {
					$text = sprintf(__('Back to "%s" attribute terms', 'woocommerce-attribute-stock'), $attribute['label']);
				} else {
					$text = __('Back to attribute terms', 'woocommerce-attribute-stock');
				}
			}
			elseif (strpos($referer, '/edit.php?post_type=product&page=product_attributes&edit=') !== false) {
				if (preg_match('/edit=(\d+)/', $referer, $matches) && isset($matches[1]) && $attribute = Attributes::get_attribute($matches[1])) {
					$text = sprintf(__('Back to "%s" attribute', 'woocommerce-attribute-stock'), $attribute['label']);
				} else {
					$text = __('Back to attribute', 'woocommerce-attribute-stock');
				}
			}
			elseif (strpos($referer, '/edit.php?post_type=product&page=product_attributes') !== false) {
				$text = __('Back to attributes', 'woocommerce-attribute-stock');
			}
			elseif (strpos($referer, '/admin.php?page=wc-reports&tab=mewz_attribute_stock') !== false) {
				$text = __('Back to stock report', 'woocommerce-attribute-stock');
			}
		}

		if (!isset($text)) {
			$text = __('Back to attribute stock', 'woocommerce-attribute-stock');
			$referer = admin_url('edit.php?post_type=mewz_attribute_stock');
		}

		return '<a href="' . esc_url($referer) . '">← ' . esc_html($text) . '</a>';
	}

	public function spoof_backlink_referer()
	{
		if (!empty($_GET['back'])) {
			$_REQUEST['_wp_http_referer'] = $_GET['back'];
		}
	}

	public function admin_body_class($class)
	{
		global $wp_version;

		if (version_compare($wp_version, '5.3', '>=') && strpos($class, 'branch-5-3') === false) {
			$class .= ' branch-5-3';
		}

	    return $class;
	}
}
