<?php
namespace Mewz\WCAS\Plugin;

use Mewz\Framework\Plugin;
use Mewz\Framework\Util\PostTypes;
use Mewz\WCAS\Actions;
use Mewz\WCAS\Compatibility;
use Mewz\WCAS\Components;
use Mewz\WCAS\Util\Matches;
use Mewz\WCAS\Util\Settings;

class Loader extends Plugin\Loader
{
	public function register_tables()
	{
		global $wpdb;

		$tables = [Matches::SETS_TABLE, Matches::ROWS_TABLE];

		foreach ($tables as $table) {
			$wpdb->$table = $wpdb->prefix . $table;
			$wpdb->tables[] = $table;
		}
	}

	public function load()
	{
		parent::load();

		// load compatibility integration overrides
		add_filter('woocommerce_integrations', [$this, 'integrations_compatibility'], 20);
	}

	public function init()
	{
		// make sure WooCommerce is active before running
		if (!defined('WC_PLUGIN_FILE')) return;

	    parent::init();

		add_action('init', [$this, 'register_post_types'], 5);
		add_action('rest_api_init', [$this, 'rest_api_init'], 5);

		// handle background tasks
		add_action('wp_ajax_mewz_wcas_task', [$this, 'ajax_handle_task']);
		add_action('wp_ajax_nopriv_mewz_wcas_task', [$this, 'ajax_handle_task']);
		add_action('shutdown', [$this, 'shutdown_dispatch_tasks'], 999999);

		// load common actions
		$this->action(Actions\Common\ClearCaches::class);
		$this->action(Actions\Common\DeleteMatches::class);
		$this->action(Actions\Common\StockNotifications::class);
		$this->action(Actions\Common\UpdateOrderStock::class);

		// load compatibility for certain plugins to play nice with attribute stock
		$this->load_compatibility_actions();

		// load admin actions
		if ($this->context->admin) {
			add_action('current_screen', [$this, 'current_screen'], 0);
			add_action('load-product_page_product_attributes', [$this, 'load_product_attributes'], 0);
			add_action('load-woocommerce_page_wc-reports', [$this, 'load_wc_reports'], 0);
			add_action('load-plugins.php', [$this, 'load_plugins_page'], 0);
		}

		// everything below this line is full version only
		if (MEWZ_WCAS_LITE) return;

		// load settings
		if ($this->context->admin) {
			$this->action(Actions\Admin\Settings\InventorySettings::class);
		}

		// load product limits
		$limit_product_stock_option = Settings::limit_product_stock();

		if ($limit_product_stock_option === 'auto') {
			$limit_product_stock = apply_filters('mewz_wcas_limit_product_stock_auto', $this->context->front || $this->context->cron || $this->context->task);
		} else {
			$limit_product_stock = $limit_product_stock_option === 'yes';
		}

		$limit_product_stock = apply_filters('mewz_wcas_limit_product_stock', $limit_product_stock, $limit_product_stock_option);

		if ($limit_product_stock) {
			$this->action(Actions\Front\ProductLimits::class);

			// load frontend actions
			if ($this->context->front) {
				add_action('wp_loaded', [$this, 'wp_loaded_front'], 0);
				add_action('wc_ajax_get_variation', [$this, 'wc_ajax_get_variation'], 0);

				$this->action(Actions\Front\ProductDisplay::class);
				$this->action(Actions\Front\ProductQuery::class);
				$this->action(Actions\Front\VariableLimits::class);
			}
		}

		// load workers
		if ($limit_product_stock || $limit_product_stock_option !== 'no') {
			if ($limit_product_stock_option === 'auto' && !class_exists(Actions\Front\ProductLimits::class, false)) {
				$this->action(Actions\Workers\AutoProductLimits::class);
			}

			$this->action(Actions\Workers\StockChange::class);
			$this->action(Actions\Workers\OutOfStock::class);
			$this->action(Actions\Workers\ProductAttributesLookup::class);
		}

		if (!class_exists(Plugin\Updater::class, false)) {
			$this->plugin->__load_updater();

			if (!class_exists(Plugin\Updater::class, false)) {
				require_once ABSPATH . 'wp-admin/includes/plugin.php';
				deactivate_plugins($this->plugin->basename);
			}
		}
	}

	public function register_post_types()
	{
		PostTypes::add('mewz_attribute_stock', [
			'show_ui' => true,
			'show_in_menu' => 'edit.php?post_type=product',
			'capability_type' => 'product',
			'map_meta_cap' => true,
			'menu_icon' => 'dashicons-clipboard',
			'supports' => ['title'],
			'description' => __('WooCommerce attribute stock items.', 'woocommerce-attribute-stock'),
			'labels' => [
				'name'          => _x('Attribute Stock', 'post type general name', 'woocommerce-attribute-stock'),
				'singular_name' => _x('Attribute Stock', 'post type singular name', 'woocommerce-attribute-stock'),
			],
		]);
	}

	public function register_scripts()
	{
		if ($this->context->admin) {
			$this->scripts->register_css('@admin/attributes');
			$this->scripts->register_js('@admin/attributes', ['translate' => true]);

			$this->scripts->register_css('@admin/stock-list');
			$this->scripts->register_js('@admin/stock-list', ['translate' => true]);

			$this->scripts->register_css('@admin/stock-edit');
			$this->scripts->register_js('@admin/stock-edit', ['translate' => true]);

			$this->scripts->register_css('@admin/reports');
		}
		elseif (!MEWZ_WCAS_LITE) {
			$this->scripts->register_js('@front/variable-stock');
		}
	}

	public function rest_api_init()
	{
		$this->action(Actions\Admin\Stock\StockAjax::class);

		if (!MEWZ_WCAS_LITE) {
			add_filter('woocommerce_rest_api_get_rest_namespaces', [$this, 'register_wc_rest_api_namespace']);
		}
	}

	public function register_wc_rest_api_namespace($namespaces)
	{
		$namespaces['wc/v3']['attribute-stock'] = Components\RestApiController::class;

	    return $namespaces;
	}

	public function admin_init()
	{
		parent::admin_init();

		if ($this->context->taxonomy && strpos($this->context->taxonomy, 'pa_') === 0) {
			$this->action(Actions\Admin\Attributes\AttributeTermEdit::class);
			$this->action(Actions\Admin\Attributes\AttributeTermSave::class);
			$this->action(Actions\Admin\Attributes\AttributeTermList::class);
		}

		$this->action(Actions\Admin\Products\ProductVariationOptions::class);

		$this->action(Actions\Admin\Stock\StockExport::class);

		if (!WP_DEBUG && !MEWZ_WCAS_LITE && !class_exists(Plugin\Authorizer::class, false)) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
			deactivate_plugins($this->plugin->basename);
		}
	}

	public function ajax_handle_task()
	{
		$this->tasks->handle($_GET, $_POST);
	}

	public function shutdown_dispatch_tasks()
	{
		$this->tasks->dispatch_tasks(!$this->context->task);
	}

	public function current_screen()
	{
		switch ($this->context->screen_id) {
			case 'edit-mewz_attribute_stock':
				$this->action(Actions\Admin\Stock\StockList::class);
				$this->action(Actions\Admin\Stock\StockListFilters::class);
				$this->action(Actions\Admin\Stock\StockListQuery::class);
				$this->action(Actions\Admin\Stock\StockActions::class);
				$this->action(Actions\Admin\Stock\StockBulkActions::class);
				$this->action(Actions\Admin\Plugin\PluginHelp::class);
				break;

			case 'mewz_attribute_stock':
				$this->action(Actions\Admin\Stock\StockEdit::class);
				$this->action(Actions\Admin\Stock\StockSave::class);
				$this->action(Actions\Admin\Stock\StockActions::class);
				$this->action(Actions\Admin\Plugin\PluginHelp::class);
				break;

			case 'product':
				$this->action(Actions\Admin\Products\ProductOptions::class);
				break;
		}
	}

	public function load_product_attributes()
	{
		$this->action(Actions\Admin\Attributes\AttributeEdit::class);
		$this->action(Actions\Admin\Attributes\AttributeSave::class);

		if (!isset($_GET['edit'])) {
			$this->action(Actions\Admin\Attributes\AttributeList::class);
		}
	}

	public function load_wc_reports()
	{
		$this->action(Actions\Admin\Reports\StockReport::class);
	}

	public function load_plugins_page()
	{
		$this->action(Actions\Admin\Plugin\PluginLinks::class);
	}

	public function wp_loaded_front()
	{
		if (WC()->cart) {
			$this->action(Actions\Front\CartAdd::class);
			$this->action(Actions\Front\CartItems::class);
		}
	}

	public function wc_ajax_get_variation()
	{
		$this->action(Actions\Front\VariableLimitsAjax::class);
	}

	public function load_compatibility_actions()
	{
		$this->action(Compatibility\Actions\WooCommerce::class);

		if (!MEWZ_WCAS_LITE) {
			// WooCommerce WPML
			if (defined('WCML_VERSION')) {
				$this->action(Compatibility\Actions\WPML::class);
			}

			// Polylang for WooCommerce
			if (defined('PLLWC_VERSION')) {
				$this->action(Compatibility\Actions\Polylang::class);
			}

			// WP Rocket
			if (defined('WP_ROCKET_VERSION')) {
				$this->action(Compatibility\Actions\WPRocket::class);
			}

			// WP-Lister (for eBay and Amazon)
			if (defined('WPLE_PLUGIN_VERSION') || defined('WPLA_VERSION')) {
				$this->action(Compatibility\Actions\WPLister::class);
			}

			// Xootix Waitlist WooCommerce
			if (defined('XOO_WL_PLUGIN_FILE')) {
				$this->action(Compatibility\Actions\XootixWaitlist::class);
			}
		}

		// Lumise Product Designer
		if (defined('LUMISE_WOO')) {
			$this->action(Compatibility\Actions\LumiseProductDesigner::class);
		}

		// WooCommerce Order Status & Actions Manager
		if (function_exists('WC_SA')) {
			$this->action(Compatibility\Actions\OrderStatusActions::class);
		}
	}

	public function integrations_compatibility($integrations)
	{
		// WooCommerce Cart Stock Reducer
		if (!MEWZ_WCAS_LITE && $this->context->front && class_exists(\WC_Cart_Stock_Reducer::class, false)) {
			$csr_index = array_search(\WC_Cart_Stock_Reducer::class, $integrations, true);

			if ($csr_index !== false) {
				$integrations[$csr_index] = Compatibility\Components\WCCartStockReducer::class;
			}
		}

		return $integrations;
	}
}
