<?php
namespace Mewz\WCAS\Util;

use Mewz\Framework\Compatibility\Multilang;

class Attributes
{
	public static function get_attributes($refresh = false)
	{
		static $attributes;

		if ($attributes === null || $refresh) {
			$attribute_list = wc_get_attribute_taxonomies();
			$attributes = [];

			foreach ($attribute_list as $attribute) {
				$id = (int)$attribute->attribute_id;
				$name = (string)$attribute->attribute_name;
				$label = (string)$attribute->attribute_label;

				$attributes[$id] = [
					'id' => $id,
					'name' => $name,
					'label' => $label !== '' ? $label : $name,
				];
			}
		}

		return $attributes;
	}

	public static function get_attribute($attribute)
	{
		$attribute_id = self::get_attribute_id($attribute);
		if (!$attribute_id) return false;

		$attributes = self::get_attributes();

		return isset($attributes[$attribute_id]) ? $attributes[$attribute_id] : false;
	}

	public static function get_attribute_id($attribute)
	{
		if (is_string($attribute) && $attribute !== (string)(int)$attribute) {
			$attributes = self::get_attributes();

			if (strpos($attribute, 'pa_') === 0) {
				$attribute = substr($attribute, 3);
			}

			$attribute = wc_sanitize_taxonomy_name($attribute);

			foreach ($attributes as $id => $attr) {
			    if ($attr['name'] === $attribute) {
			    	return $id;
			    }
			}

			return false;
		}

		return (int)$attribute;
	}

	public static function get_attribute_name($attribute, $get_taxonomy = false)
	{
		if (is_string($attribute) && $attribute !== (string)(int)$attribute) {
			$attribute = wc_sanitize_taxonomy_name($attribute);
			$is_taxonomy = strpos($attribute, 'pa_') === 0;

			if ($get_taxonomy === $is_taxonomy) {
				return $attribute;
			} else {
				return $get_taxonomy ? 'pa_' . $attribute : substr($attribute, 3);
			}
		} else {
			$attributes = self::get_attributes();

			if (!isset($attributes[$attribute])) {
				return false;
			}

			$attribute_name = $attributes[$attribute]['name'];

			if ($get_taxonomy) {
				$attribute_name = 'pa_' . $attribute_name;
			}

			return $attribute_name;
		}
	}

	public static function get_attribute_label($attribute)
	{
		$taxonomy = self::get_attribute_name($attribute, true);
		if (!$taxonomy) return false;

		return wc_attribute_label($taxonomy);
	}

	public static function get_attribute_id_sets($attribute_data, $bypass_multilang = true)
	{
		$sets = [];

		if ($bypass_multilang) {
			Multilang::toggle_term_filters(false);
		}

		foreach ($attribute_data as $attribute => $terms) {
			$attribute_id = self::get_attribute_id($attribute);
			$taxonomy = null;

			foreach ((array)$terms as $term) {
				if (is_string($term)) {
					if ($taxonomy === null) {
						$taxonomy = self::get_attribute_name($attribute_id, true);
					}

					$term = get_term_by('slug', $term, $taxonomy);

					$term_id = $term ? $term->term_id : 0;
				} else {
					$term_id = (int)$term;
				}

				if ($bypass_multilang && $term_id > 0) {
					if ($taxonomy === null) {
						$taxonomy = self::get_attribute_name($attribute_id, true);
					}

					$term_id = Multilang::get_translated_object_id($term_id, 'term', $taxonomy, 'default');
				}

				$sets[$attribute_id][$term_id] = $term_id;
			}

			if (!empty($sets[$attribute_id])) {
				sort($sets[$attribute_id]);
			}
		}

		if ($bypass_multilang) {
			Multilang::toggle_term_filters(true);
		}

		ksort($sets);

		return $sets;
	}

	public static function sluggify_attribute_terms($attributes)
	{
		$slugged = [];

		foreach ($attributes as $key => $terms) {
			if (!is_array($terms)) continue;

			foreach ($terms as $term) {
				if (is_string($term)) {
					$slugged[$key][] = $term;
				} else {
					$slug = self::get_term_prop($term, wc_sanitize_taxonomy_name($key), 'slug');

					if ($slug !== false) {
						$slugged[$key][] = $slug;
					}
				}
			}
		}

		return $slugged;
	}

	public static function get_term_prop($term_id, $taxonomy, $prop)
	{
		static $cache = [];

		if (isset($cache[$taxonomy][$term_id][$prop])) {
			return $cache[$taxonomy][$term_id][$prop];
		}

		$term = get_term($term_id, $taxonomy);

		if (!$term || $term instanceof \WP_Error) {
			return $cache[$taxonomy][$term_id][$prop] = false;
		}

		return $cache[$taxonomy][$term_id][$prop] = $term->$prop;
	}

	public static function has_catchall(array $attributes)
	{
		foreach ($attributes as $term) {
			if ($term === '' || $term === 0) {
				return true;
			}
		}

		return false;
	}

	public static function strip_attribute_prefix($attributes)
	{
		if (empty($attributes) || strpos(key($attributes), 'attribute_') !== 0) {
			return $attributes;
		}

		$stripped = [];

		foreach ($attributes as $attr => $term) {
			$stripped[substr($attr, 10)] = $term;
		}

	    return $stripped;
	}

	public static function get_request_attributes()
	{
		if (empty($_REQUEST)) return false;

		$attributes = [];

		foreach ($_REQUEST as $key => $value) {
			if (strpos($key, 'attribute_') === 0) {
				$attributes[substr($key, 10)] = stripslashes($value);
			}
		}

		return $attributes;
	}

	public static function decode_keys(array $attributes)
	{
	    $decoded = [];

		foreach ($attributes as $key => $value) {
			$decoded[wc_sanitize_taxonomy_name($key)] = $value;
		}

		return $decoded;
	}

	public static function encode_keys(array $attributes)
	{
		$encoded = [];

		foreach ($attributes as $key => $value) {
			$encoded[sanitize_title($key)] = $value;
		}

		return $encoded;
	}
}
