<?php
namespace Mewz\WCAS\Util;

use Mewz\Framework\Util\Number;
use Mewz\QueryBuilder\DB;
use Mewz\WCAS\Models\AttributeStock;

class Export
{
	const FIELDS = [
		'title' => '',
		'sku' => '',
		'quantity' => '',
		'notes' => '',
		'low_stock' => '',
		'limit_products' => '',
		'match_all' => '',
		'enabled' => '',
		'products' => '',
		'exclude_products' => '',
		'categories' => '',
		'exclude_categories' => '',
		'product_types' => '',
		'attributes' => '',
	];

	public static function to_csv_download(array $stock_ids)
	{
		$filename = 'attribute-stock-' . current_time('Y-m-d-His') . '.csv';
		$filename = apply_filters('mewz_wcas_export_filename', $filename, $stock_ids);

		$fields = apply_filters('mewz_wcas_export_fields', self::FIELDS, $stock_ids);

		// if headers have already been sent, there's not much else we can do here
		if (headers_sent()) return false;

		header('Content-Type: text/csv');
		header('Content-Disposition: attachment; filename="' . $filename . '"');

		$out = fopen('php://output', 'w');

		fputcsv($out, array_keys($fields));

		wp_ob_end_flush_all();
		flush();

		$row_count = 0;

		foreach ($stock_ids as $stock_id) {
			if ($row = self::build_export_row($fields, $stock_id)) {
				fputcsv($out, $row);

				if (++$row_count % 10 === 0) {
					wp_ob_end_flush_all();
					flush();
				}
			}
		}

		fclose($out);
		die;
	}

	/**
	 * @param array $row
	 * @param int $stock_id
	 *
	 * @return mixed
	 */
	public static function build_export_row(array $row, $stock_id)
	{
		$stock = new AttributeStock($stock_id, 'edit');

		if (isset($row['attributes']) && $attributes = self::build_attributes_export_data($stock_id)) {
			$row['attributes'] = json_encode($attributes);
		}

		$props = array_diff(array_keys($row), [
			'products',
			'exclude_products',
			'categories',
			'exclude_categories',
			'attributes',
		]);

		foreach ($props as $prop) {
			if (!method_exists($stock, $prop)) {
				continue;
			}

			$value = $stock->$prop();

			if (is_array($value)) {
				$row[$prop] = implode(', ', $value);
			} elseif (is_bool($value)) {
				$row[$prop] = $value ? 'yes' : 'no';
			} elseif (is_float($value)) {
				$row[$prop] = Number::period_decimal($value);
			} else {
				$row[$prop] = (string)$value;
			}
		}

		isset($row['products']) && $row['products'] = self::build_product_list($stock->products());
		isset($row['exclude_products']) && $row['exclude_products'] = self::build_product_list($stock->exclude_products());

		isset($row['categories']) && $row['categories'] = self::build_category_list($stock->categories());
		isset($row['exclude_categories']) && $row['exclude_categories'] = self::build_category_list($stock->exclude_categories());

		if (isset($row['attributes']) && $attributes = self::build_attributes_export_data($stock_id)) {
			$row['attributes'] = json_encode($attributes);
		}

		return apply_filters('mewz_wcas_export_row', $row, $stock_id);
	}

	public static function build_product_list($product_ids)
	{
		$list = [];

		foreach ($product_ids as $product_id) {
			$product = wc_get_product($product_id);
			if (!$product) continue;

			$ident = $product->get_sku('edit');

			if ($ident == '') {
				$ident = $product->get_id() . '-' . $product->get_slug('edit');
			}

			$list[] = $ident;
		}

		return implode(', ', $list);
	}

	public static function build_category_list($category_ids)
	{
		$list = [];

		foreach ($category_ids as $category_id) {
			$category = get_term($category_id);

			if (!$category || is_wp_error($category)) {
				continue;
			}

			$list[] = $category->slug;
		}

		return implode(', ', $list);
	}

	/**
	 * @param int $stock_id
	 *
	 * @return array
	 */
	public static function build_attributes_export_data($stock_id)
	{
		$match_sets = Matches::get_sets($stock_id);
		$attribute_list = Attributes::get_attributes();

		$attributes = [];

		foreach ($match_sets as $set) {
			$attribute = ['attr' => []];

			if ($set['multiplier'] != 1) {
				$attribute['x'] = $set['multiplier'];
			}

			foreach ($set['rows'] as $row) {
				if (!isset($attribute_list[$row['attribute']])) {
					continue;
				}

				if ($row['term']) {
					$term = get_term($row['term']);

					if (!$term || is_wp_error($term)) {
						continue;
					}

					$term_slug = $term->slug;
				} else {
					$term_slug = '';
				}

				$attribute['attr'][] = [
					$attribute_list[$row['attribute']]['name'],
					$term_slug,
				];
			}

			if ($attribute['attr']) {
				$attributes[] = $attribute;
			}
		}

		return $attributes;
	}

	/**
	 * @param AttributeStock $stock
	 *
	 * @return array
	 */
	public static function build_filters_export_data($stock)
	{
		$filters = [];

		$product_filters = [
			'incl' => $stock->products(),
			'excl' => $stock->exclude_products(),
		];

		foreach ($product_filters as $type => $product_ids) {
		    foreach ($product_ids as $product_id) {
		    	$sku = get_post_meta($product_id, '_sku', true);
			    $filters['products'][$type][] = $sku != '' ? $sku : (int)$product_id;
		    }
		}

		$category_filters = [
			'incl' => $stock->categories(),
			'excl' => $stock->exclude_categories(),
		];

		foreach ($category_filters as $type => $cat_ids) {
			foreach ($cat_ids as $cat_id) {
				$term = get_term($cat_id);

				if (!$term || is_wp_error($term)) {
					continue;
				}

				$filters['categories'][$type][] = $term->slug;
			}
		}

	    if ($product_types = $stock->product_types()) {
		    $filters['product_types'] = $product_types;
	    }

		return $filters;
	}

	public static function import_row($row)
	{
		$row = apply_filters('mewz_wcas_import_row', $row);

		if (!$row) {
			return false;
		}

		$data = [];

		foreach ($row as $key => $value) {
			if ($value === '' || $key === 'attributes') {
				continue;
			}

			if ($value === '()') {
				$value = '';

				if (in_array($key, ['title', 'sku'])) {
					continue;
				}
			}

			$data[$key] = $value;
		}

		$attributes = null;

		if (!empty($row['attributes'])) {
			if ($row['attributes'] === '()') {
				$attributes = [];
			} elseif ($attributes = json_decode($row['attributes'], true)) {
				$attributes = self::match_attributes($attributes);
			}
		}

		if (isset($data['sku']) || isset($data['title'])) {
			$stock = self::match_stock_item($data);
			$update = $stock->exists();

			if (!$update && !isset($data['title'])) {
				$data['title'] = $data['sku'];
			}
		} elseif ($attributes) {
			$data['title'] = self::attributes_to_title($attributes);
			$stock = new AttributeStock(null, 'edit');
			$update = false;
		} else {
			// no identifier fields to import
			return false;
		}

		foreach (['match_all', 'limit_products', 'enabled'] as $key) {
			if (isset($data[$key])) {
				$data[$key] = wc_string_to_bool($data[$key]);
			}
		}

		isset($data['products']) && $data['products'] = self::match_products($data['products']);
		isset($data['exclude_products']) && $data['exclude_products'] = self::match_products($data['exclude_products']);

		isset($data['categories']) && $data['categories'] = self::match_categories($data['categories']);
		isset($data['exclude_categories']) && $data['exclude_categories'] = self::match_categories($data['exclude_categories']);

		isset($data['product_types']) && $data['product_types'] = self::list_to_array($data['product_types']);

		$stock->bind($data);

		if ($stock->save() === false) {
			return false;
		}

		if (is_array($attributes)) {
			Matches::save_sets($stock->id(), $attributes);
		}

		do_action('mewz_wcas_imported_row', $row, $stock, $update);

		return $update ? 'updated' : 'added';
	}

	public static function match_stock_item($data)
	{
		$query = DB::table('posts', 'p')
			->where('p.post_type', 'mewz_attribute_stock')
			->where('p.post_status', ['publish', 'draft']);

		if (isset($data['sku'])) {
			$query->left_join('postmeta', 'sku')->on("sku.post_id = p.ID AND sku.meta_key = '_sku'");
		}

		if (isset($data['sku'], $data['title'])) {
			$query->where('sku.meta_value = ? OR p.post_title = ?', $data['sku'], $data['title']);
		}
		elseif (isset($data['sku'])) {
			$query->where('sku.meta_value', $data['sku']);
		}
		elseif (isset($data['title'])) {
			$query->where('p.post_title', $data['title']);
		}

		$stock_id = $query->val('p.ID') ?: null;

		return new AttributeStock($stock_id, 'edit');
	}

	public static function match_products($list)
	{
		$idents = self::list_to_array($list);
		$product_ids = [];

		foreach ($idents as $ident) {
			$product_id = wc_get_product_id_by_sku($ident);

			if (!$product_id) {
				$post = get_page_by_path($ident, OBJECT, 'product') ?: get_page_by_path($ident, OBJECT, 'product_variation');

				if ($post) {
					$product_id = $post->ID;
				}
			}

			if (!$product_id) {
				$parts = explode('-', $ident, 2);

				if ($parts[0] === (string)(int)$parts[0]) {
					if (isset($parts[1]) && $parts[1] != '') {
						$post = get_page_by_path($parts[1], OBJECT, 'product') ?: get_page_by_path($parts[1], OBJECT, 'product_variation');

						if ($post) {
							$product_id = $post->ID;
						}
					}

					if (!$product_id && in_array(get_post_type($parts[0]), ['product', 'product_variation'])) {
						$product_id = (int)$parts[0];
					}
				}
			}

			if ($product_id > 0) {
				$product_ids[] = $product_id;
			}
		}

		return $product_ids;
	}

	public static function match_categories($list)
	{
		$cat_slugs = self::list_to_array($list);
		$cat_ids = [];

		foreach ($cat_slugs as $cat_slug) {
		    $cat_term = get_term_by('slug', $cat_slug, 'product_cat');

		    if ($cat_term && !is_wp_error($cat_term)) {
			    $cat_ids[] = $cat_term->term_id;
		    }
		}

		return $cat_ids;
	}

	public static function match_attributes($attributes)
	{
		$sets = [];

		foreach ($attributes as $group) {
			$set['rows'] = [];

			foreach ($group['attr'] as $row) {
				$attr_id = Attributes::get_attribute_id($row[0]);
				if (!$attr_id) continue;

				if ($row[1] !== '') {
					$term = get_term_by('slug', $row[1], 'pa_' . $row[0]);
					$term_id = $term && !is_wp_error($term) ? $term->term_id : 0;
				} else {
					$term_id = 0;
				}

				$set['rows'][] = [
					'attribute' => $attr_id,
					'term' => $term_id,
				];
			}

			if (isset($group['x'])) {
				$set['multiplier'] = $group['x'];
			}

			$sets[] = $set;
		}

		return $sets;
	}

	public static function attributes_to_title($attributes)
	{
		$attr = [];

		foreach ($attributes as $group) {
		    foreach ($group['rows'] as $row) {
		    	if (isset($attr[$row['attribute']])) {
				    $attr[$row['attribute']][] = $row['term'];
		    	} else {
				    $attr[$row['attribute']] = [$row['term']];
		    	}
		    }
		}

		$title = [];

		foreach ($attr as $attr_id => $term_ids) {
			$title_attr = Attributes::get_attribute_label($attr_id);

			$title_terms = [];
			$attr_taxonomy = Attributes::get_attribute_name($attr_id, true);

			foreach ($term_ids as $term_id) {
			    $term = get_term($term_id, $attr_taxonomy);
				$title_terms[] = $term->name;
			}

			$title[] = $title_attr . ': ' . implode('|', $title_terms);
		}

		return implode(', ', $title);
	}

	public static function list_to_array($list)
	{
		$items = [];

		foreach (explode(',', $list) as $item) {
			$item = trim($item);

			if ($item !== '') {
				$items[] = $item;
			}
		}

		return $items;
	}
}
