<?php
namespace Mewz\Framework\Services;

use Mewz\Framework\Compatibility;
use Mewz\Framework\Plugin;

class Scripts
{
	/** @var Plugin */
	protected $plugin;

	/** @var string Prefix to add to handles starting with "@" */
	protected $prefix;

	/** @var string Language directory path */
	protected $language_path;

	/** @var array The queue of scripts to enqueue later */
	protected $queue = [];

	/** @var bool Set to true when `enqueue_scripts` action fires and scripts queue is processed */
	protected $enqueue_scripts = false;

	/**
	 * @param Plugin $plugin
	 * @param string $prefix
	 */
	public function __construct(Plugin $plugin, $prefix = null, $language_path = null)
	{
		$this->plugin = $plugin;
		$this->prefix = $prefix !== null ? $prefix . '-' : '';
		$this->language_path = $language_path === null ? $plugin->base_dir . '/languages' : null;
	}

	/**
	 * @return string
	 */
	public function get_prefix()
	{
		return $this->prefix;
	}

	/**
	 * @return array
	 */
	public function get_queue()
	{
		return $this->queue;
	}

	/**
	 * @return bool
	 */
	public function enqueue_scripts()
	{
		return $this->enqueue_scripts;
	}

	/**
	 * @param string $type 'css' or 'js'
	 * @param string $handle Unique script handle. First char "@" will be replaced with plugin prefix.
	 * @param mixed ...$options Array of key/value pairs, optionally following a $path string.
	 *                          Keys: path, deps, ver, media, in_footer, min, translate
	 *
	 * @return bool
	 */
	public function register($type, $handle, ...$options)
	{
		$args = $this->expand_args($type, $handle, $options);

		if ($type === 'css') {
			return wp_register_style($args['handle'], $args['path'], $args['deps'], $args['ver'], $args['media']);
		}
		elseif ($type === 'js') {
			$result = wp_register_script($args['handle'], $args['path'], $args['deps'], $args['ver'], $args['in_footer']);

			if ($result && !empty($args['translate'])) {
				Compatibility\I18n::set_script_translations($args['handle'], $args['translate'], $this->language_path);
			}

			return $result;
		}

		return false;
	}

	/**
	 * @param string $handle Unique script handle. First char "@" will be replaced with plugin prefix.
	 * @param mixed ...$options Array of key/value pairs, optionally following a $path string.
	 *                          Keys: path, deps, ver, media, in_footer, min, translate
	 *
	 * @return bool
	 */
	public function register_css($handle, ...$options)
	{
		return $this->register('css', $handle, ...$options);
	}

	/**
	 * @param string $handle Unique script handle. First char "@" will be replaced with plugin prefix.
	 * @param mixed ...$options Array of key/value pairs, optionally following a $path string.
	 *                          Keys: path, deps, ver, media, in_footer, min, translate
	 *
	 * @return bool
	 */
	public function register_js($handle, ...$options)
	{
		return $this->register('js', $handle, ...$options);
	}

	/**
	 * @param string $type 'css' or 'js'
	 * @param string $handle Unique script handle. First char "@" will be replaced with plugin prefix.
	 * @param mixed ...$options Array of key/value pairs, optionally following a $path string.
	 *                          Keys: path, deps, ver, media, in_footer, min, translate
	 */
	public function enqueue($type, $handle, ...$options)
	{
		if (!$this->enqueue_scripts) {
			$this->queue[$type][$handle] = $options;
			return;
		}

		if ($options) {
			$args = $this->expand_args($type, $handle, $options);

			if ($type === 'css') {
				wp_enqueue_style($args['handle'], $args['path'], $args['deps'], $args['ver'], $args['media']);
			}
			elseif ($type === 'js') {
				wp_enqueue_script($args['handle'], $args['path'], $args['deps'], $args['ver'], $args['in_footer']);

				if (!empty($args['translate'])) {
					$this->inline_translations($args['handle'], $args['translate']);
				}
			}
		} else {
			$handle = $this->expand_handle($handle);

			if ($type === 'css') {
				wp_enqueue_style($handle);
			} elseif ($type === 'js') {
				wp_enqueue_script($handle);
				$this->inline_translations($handle);
			}
		}
	}

	/**
	 * @param string $handle Unique script handle. First char "@" will be replaced with plugin prefix.
	 * @param mixed ...$options Array of key/value pairs, optionally following a $path string.
	 *                          Keys: path, deps, ver, media, in_footer, min, translate
	 */
	public function enqueue_css($handle, ...$options)
	{
		$this->enqueue('css', $handle, ...$options);
	}

	/**
	 * @param string $handle Unique script handle. First char "@" will be replaced with plugin prefix.
	 * @param mixed ...$options Array of key/value pairs, optionally following a $path string.
	 *                          Keys: path, deps, ver, media, in_footer, min, translate
	 */
	public function enqueue_js($handle, ...$options)
	{
		$this->enqueue('js', $handle, ...$options);
	}

	/**
	 * @param string $handle Unique script handle. First char "@" will be replaced with plugin prefix.
	 * @param mixed ...$options Array of key/value pairs, optionally following a $path string.
	 *                          Keys: path, deps, ver, media, in_footer, min, translate
	 */
	public function enqueue_bundle($handle, ...$options)
	{
		$this->enqueue('css', $handle, ...$options);
		$this->enqueue('js', $handle, ...$options);
	}

	/**
	 * Enqueue all scripts in the queue.
	 *
	 * @see Scripts::enqueue()
	 */
	public function process_queue()
	{
		$this->enqueue_scripts = true;

		if ($this->queue) {
			foreach ($this->queue as $type => $handles) {
				foreach ($handles as $handle => $options) {
					$this->enqueue($type, $handle, ...$options);
				}
			}
		}
	}

	/**
	 * Alias for {@see wp_localize_script()}.
	 *
	 * @param string $handle Unique script handle. First char "@" will be replaced with plugin prefix.
	 * @param string $object_name Name for the JavaScript object. Passed directly, so it should be a qualified JS variable.
	 * @param array $data The data itself. The data can be either a single or multi-dimensional array.
	 *
	 * @return bool
	 */
	public function export_js($handle, $object_name, $data)
	{
		return wp_localize_script($this->expand_handle($handle), $object_name, $data);
	}

	/**
	 * Alias for {@see wp_set_script_translations()} with backward compatibility.
	 *
	 * @param string $handle
	 * @param string $domain
	 * @param string $path
	 */
	public function set_translations($handle, $domain = null, $path = null)
	{
		$handle = $this->expand_handle($handle);

		if ($domain === null) {
			$domain = $this->plugin->domain;
		}

		if ($path === null) {
			$path = $this->language_path;
		}

		Compatibility\I18n::set_script_translations($handle, $domain, $path);
	}

	/**
	 * Internal helper to inline translations for enqueued scripts when native JS translations
	 * aren't supported, i.e. WP < 5.0. If natively supported, this does nothing.
	 *
	 * @param string $handle The already-expanded script handle
	 * @param string $domain If not set, looks for translation domain set at script registration
	 */
	protected function inline_translations($handle, $domain = null)
	{
		if (Compatibility\I18n::native_js_i18n()) {
			return;
		}

		$wp_scripts = wp_scripts();

		$data = $wp_scripts->get_data($handle, 'mewz-translate');
		$path = $data['path'];

		if ($domain === null) {
			$domain = $data['domain'];
		}

		if (!$domain) return;

		$json = Compatibility\I18n::load_script_textdomain($handle, $domain, $path) ?: 'null';

		$output = "mewz.loadTranslations('$domain', $json);";

		$wp_scripts->add_inline_script($handle, $output, 'before');
	}

	/**
	 * @param string $handle
	 *
	 * @return string
	 */
	public function expand_handle($handle)
	{
		if ($handle[0] === '@') {
			$handle = $this->prefix . substr($handle, 1);
		}

		return $handle;
	}

	/**
	 * @param string $type
	 * @param string $handle
	 * @param array $options
	 *
	 * @return array
	 */
	public function expand_args($type, $handle, array $options = [])
	{
		$args = [
			'handle' => $this->expand_handle($handle),
			'path' => null,
			'deps' => [],
			'ver' => null,
			'min' => '.min',
		];

		if ($type === 'css') {
			$args['media'] = 'all';
		} elseif ($type === 'js') {
			$args['deps'] = ['jquery'];
			$args['in_footer'] = true;
		}

		if ($options) {
			if (is_string($options[0])) {
				$args['path'] = $options[0];

				if (isset($options[1])) {
					$args = array_merge($args, $options[1]);
				}
			} else {
				$args = array_merge($args, $options[0]);
			}

			if ($args['path'] !== null) {
				if (SCRIPT_DEBUG && $args['min']) {
					$args['path'] = str_replace($args['min'] . '.', '.', $args['path']);
				}

				if (strpos($args['path'], '//') === false) {
					$args['path'] = $this->plugin->assets->dist($args['path'] . '.' . $type);
				}
			}

			if (!empty($args['deps'])) {
				$args['deps'] = (array)$args['deps'];

				foreach ($args['deps'] as &$dep) {
					$dep = $this->expand_handle($dep);
				}
			}

			if (!empty($args['translate'])) {
				if ($args['translate'] === true) {
					$args['translate'] = $this->plugin->domain;
				}

				$args['deps'][] = 'mewz-i18n';
			}
		}

		if ($args['path'] == null && $handle !== $args['handle']) { // handle was expanded, i.e. this is a plugin script
			$args['path'] = substr($handle, 1);

			if (!SCRIPT_DEBUG && $args['min']) {
				$args['path'] .= $args['min'];
			}

			$args['path'] = $this->plugin->assets->dist($args['path'] . '.' . $type);
		}

		if ($args['ver'] === null) {
			$args['ver'] = $this->plugin->version;
		}

		return $args;
	}
}
