<?php
namespace Mewz\WCAS\Actions\Front;

use Mewz\Framework\Base\Action;
use Mewz\Framework\Util\WooCommerce;
use Mewz\WCAS\Util\Attributes;
use Mewz\WCAS\Util\Limits;
use Mewz\WCAS\Util\Matches;
use Mewz\WCAS\Util\Products;
use Mewz\WCAS\Util\Settings;

class VariableLimits extends Action
{
	public static $templates = [
		'single-product/add-to-cart/variable.php',
		'single-product/bundled-product-variable.php',
	];

	public $js_data = [];
	public $after_template;

	public function __hooks()
	{
		add_action('woocommerce_before_template_part', [$this, 'before_template_part'], 10, 4);
		add_action('wp_print_footer_scripts', [$this, 'output_js_data'], 0, 0);
	}

	public function output_js_data($ajax = false)
	{
		$js_data = $this->get_js_data();
		if (!$js_data) return;

		if (!$ajax) {
			$this->scripts->enqueue_js('@front/variable-stock');
			$this->scripts->export_js('@front/variable-stock', 'mewz_wcas_variable_data', $js_data);
		} else {
			// output js script and data within ajax rendered template so it can be executed dynamically
			$this->view->render('front/variable-js-data-script', [
				'js_data' => $js_data,
				'inline_script' => file_get_contents($this->assets->dir('dist/front/variable-stock' . (SCRIPT_DEBUG ? '' : '.min') . '.js')),
			]);
		}
	}

	public function get_js_data()
	{
		$js_data = $this->js_data;

		if (!isset($js_data['settings'])) {
			$js_data['settings'] = [
				'no_stock_amount' => WooCommerce::no_stock_amount(),
				'low_stock_amount' => WooCommerce::low_stock_amount(),
				'hide_out_of_stock' => WooCommerce::hide_out_of_stock(),
				'outofstock_variations' => Settings::outofstock_variations(),
				'unmatched_any_variations' => Settings::unmatched_any_variations(),
			];
		}

		// don't hate on the short keys - every byte counts!
		$js_data = apply_filters('mewz_wcas_variable_js_data', $js_data);

		if (!$js_data || empty($js_data['data'])) {
			return false;
		}

		return $js_data;
	}

	public function before_template_part($template, $path, $located, $args)
	{
		if (!in_array($template, self::$templates, true)) {
			return;
		}

		if (!empty($args['available_variations'])) {
			$variations = $args['available_variations'];
		}
		elseif (!empty($args['bundled_product_variations'])) {
			if ($args['bundled_item']->use_ajax_for_product_variations()) {
				return;
			}

			$variations = $args['bundled_product_variations'];
		}
		else return;

		if (isset($args['bundled_product'])) {
			$product = $args['bundled_product'];
		} else {
			/** @var \WC_Product_Variable $product */
			global $product;
		}

		if (!$product) return;

		if (isset($this->js_data['products'][$product->get_id()]) || !$product instanceof \WC_Product_Variable || Products::is_product_excluded($product)) {
			return;
		}

		$hide_out_of_stock = WooCommerce::hide_out_of_stock() || in_array(Settings::outofstock_variations(), ['greyedout', 'hidden']);
		$display_stock = WooCommerce::stock_format() !== 'no_amount';

		$cache_key = 'variable_js_data_' . (int)$hide_out_of_stock . '_' . (int)$display_stock;
		$cache_tags = ['match_sets', 'stock', 'product_' . $product->get_id()];

		$js_data = $this->cache->get($cache_key, $cache_tags);

		if ($js_data === null) {
			$variations_data = [];
			$variations_attr = [];
			$oos_variations_count = 0;

			foreach ($variations as $variation) {
				if (empty($variation['is_in_stock'])) {
					$oos_variations_count++;
					continue;
				}

				if (!Attributes::has_catchall($variation['attributes'])) {
					continue;
				}

				$variation_id = $variation['variation_id'];
				$product_variation = wc_get_product($variation_id);

				if (!$product_variation || ($stock_status = $product_variation->get_stock_status()) === 'onbackorder' || Products::is_product_excluded($product_variation, false)) {
					continue;
				}

				$variation_data = [];

				if ($display_stock) {
					$product_variation->mewz_wcas_bypass_limits = true;
					$managing_stock = $product_variation->get_manage_stock();

					if ($managing_stock || $stock_status === 'instock') {
						$product_variation->mewz_wcas_allow_backorders_override = true;
						$variation_data['b'] = $product_variation->get_backorders();

						if ($managing_stock) {
							$variation_data['q'] = $product_variation->get_stock_quantity();
						}
					}

					$product_variation->mewz_wcas_bypass_limits = false;
				}

				$variations_data[$variation_id] = $variation_data;
				$product_variations[$variation_id] = $product_variation;

				foreach ($variation['attributes'] as $attr => $term) {
					if ($term === '') {
						$variations_attr[$attr] = '';
					} elseif (!isset($variations_attr[$attr]) || $variations_attr[$attr] !== '') {
						$variations_attr[$attr][$term] = $term;
					}
				}
			}

			if (empty($variations_data) || empty($product_variations) || empty($variations_attr)) {
				$this->cache->set($cache_key, [], $cache_tags);
				return;
			}

			$variations_attr = Attributes::strip_attribute_prefix($variations_attr);

			$product_attr = Products::get_product_attributes($product, null, true) ?: [];
			$product_attr = Attributes::encode_keys($product_attr);

			foreach ($variations_attr as $attr => $terms) {
				if ($terms === '') {
					if (isset($product_attr[$attr]) && is_array($product_attr[$attr])) {
						$variations_attr[$attr] = $product_attr[$attr];
					}
				} elseif (is_array($terms)) {
					$variations_attr[$attr] = array_values($terms);
				}

				unset($product_attr[$attr]);
			}

			$all_attr = $variations_attr + $product_attr;
			$match_data = Matches::get_all_match_data($product_variations, $all_attr, true);

			if (!$match_data || empty($match_data['matches'])) {
				// no valid attribute stock items were matched
				$this->cache->set($cache_key, [], $cache_tags);
				return;
			}

			if (($hide_out_of_stock) && $match_data['max_quantity'] <= 0 && count($variations_data) + $oos_variations_count === count($variations) && $this->compare_attributes($match_data['attributes'], $all_attr)) {
				// all variations are out of stock
				$js_data = 0;
			} else {
				$variations_attr = Attributes::sluggify_attribute_terms($variations_attr);
				$product_attr = Attributes::sluggify_attribute_terms($product_attr);

				// filter attributes if not hiding out of stock or if too many combinations
				// (it's not perfect but it's the best compromise we can do here)
				$filter_attributes = !$hide_out_of_stock;

				if (!$filter_attributes) {
					$total_combinations = $this->calculate_total_combinations($variations_attr, $product_attr);
					$filter_attributes = $total_combinations >= apply_filters('mewz_wcas_any_variation_combination_threshold', 300, $product);
				}

				if ($filter_attributes) {
					$variations_attr = $this->filter_attributes($variations_attr, $match_data['attributes']);
					$product_attr = $this->filter_attributes($product_attr, $match_data['attributes']);
				}

				$js_data = [
					'va' => $variations_attr,
					'pa' => $product_attr,
					'md' => $match_data['matches'],
					'vd' => $variations_data,
				];
			}

			$this->cache->set($cache_key, $js_data, $cache_tags);
		}

		if ($js_data === 0) {
			// there are no variations in stock and "hide out of stock items" is enabled
			// so we need to short circuit the template output to show the product as out of stock
			$this->after_template = $template;
			add_action('woocommerce_after_template_part', [$this, 'after_template_part_override'], -100, 4);
			ob_start();
		}
		elseif (!empty($js_data)) {
			if ($display_stock) {
				$js_data['sh'] = Limits::get_variable_stock_html($product);
			}

			$this->js_data['data'][$product->get_id()] = $js_data;

			if ($this->context->ajax_or_rest) {
				// the product form is being loaded via ajax, so we need to include variation data dynamically in the output
				$this->after_template = $template;
				add_action('woocommerce_after_template_part', [$this, 'after_template_part_jsdata'], -100, 4);
			}
		}
	}

	public function after_template_part_override($template, $path, $located, $args)
	{
		if ($template !== $this->after_template) {
			return;
		}

		$this->after_template = null;

		ob_end_clean();
		remove_action('woocommerce_after_template_part', [$this, 'after_template_part_override'], -100);

		if ($template === 'single-product/bundled-product-variable.php') {
			wc_get_template('single-product/bundled-product-unavailable.php', [
				'bundled_item' => $args['bundled_item'],
				'bundle' => $args['bundle'],
				'custom_product_data' => apply_filters('woocommerce_bundled_product_custom_data', [
					'is_unavailable' => 'yes',
					'is_out_of_stock' => 'yes',
					'is_required' => $args['bundled_item']->get_quantity('min', ['check_optional' => true]) > 0 ? 'yes' : 'no',
				], $args['bundled_item']),
			], false, WC_PB()->plugin_path() . '/templates/');
		} else {
			$args['available_variations'] = [];
			wc_get_template($template, $args);
		}
	}

	public function after_template_part_jsdata($template, $path, $located, $args)
	{
		if ($template !== $this->after_template) {
			return;
		}

		$this->after_template = null;
		remove_action('woocommerce_after_template_part', [$this, 'after_template_part_jsdata'], -100);

		$this->output_js_data(true);

		unset($this->js_data['data']);
	}

	public function compare_attributes($attributes, $compare)
	{
		foreach ($compare as $key => $terms) {
			if (count($attributes[$key]) < count($terms)) {
				return false;
		    }
		}

		return true;
	}

	public function filter_attributes($attributes, $filter)
	{
		$filtered = [];

		foreach ($attributes as $taxonomy => $terms) {
			if (empty($filter[$taxonomy])) continue;

			if (isset($filter[$taxonomy][''])) {
				$filtered[$taxonomy] = $terms;
				continue;
			}

			foreach ($terms as $term) {
				if (isset($filter[$taxonomy][$term])) {
					$filtered[$taxonomy][] = $term;
				}
			}
		}

		return $filtered;
	}

	public function calculate_total_combinations(...$lists)
	{
	    $total = 1;

	    foreach ($lists as $list) {
		    foreach ($list as $terms) {
			    if (!empty($terms)) {
				    $total *= count($terms);
			    }
		    }
	    }

	    return $total;
	}
}
