<?php
namespace Mewz\WCAS\Models;

use Mewz\Framework\Base\PostModel;
use Mewz\Framework\Util\Number;
use Mewz\WCAS\Util\Matches;

class AttributeStock extends PostModel
{
	const MODEL_TYPE = 'attribute_stock';
	const POST_TYPE = 'mewz_attribute_stock';

	protected static $alias_props = [
		'notes' => 'content',
	];

	protected $data = [
		'sku' => '',
		'quantity' => null,
		'low_stock' => null,
		'limit_products' => true,
		'match_all' => false,
		'products' => [],
		'exclude_products' => [],
		'categories' => [],
		'exclude_categories' => [],
		'product_types' => [],
	];

	/**
	 * @return bool
	 */
	public function enabled()
	{
		return $this->status() === 'publish';
	}

	/**
	 * @param bool $value
	 */
	public function set_enabled($value = true)
	{
		$this->set_status($value ? 'publish' : 'draft');
	}

	/**
	 * @return string
	 */
	public function sku()
	{
		return $this->get('sku');
	}

	/**
	 * @param string $value
	 */
	public function set_sku($value)
	{
		$this->set('sku', $value);
	}

	/**
	 * @return int|float
	 */
	public function quantity()
	{
		$value = $this->get('quantity');

		if ($this->context === 'view') {
			$value = (float)$value;
		}

		return $value;
	}

	/**
	 * @param int|float $value
	 */
	public function set_quantity($value)
	{
		if ($value !== null) {
			if (is_string($value)) {
				$value = Number::period_decimal($value);
			}

			$value = (float)$value;
		}

		$this->set('quantity', $value);
	}

	/**
	 * @param int|float $amount
	 *
	 * @return float New quantity
	 */
	public function adjust_quantity($amount)
	{
		if (is_string($amount)) {
			$amount = Number::period_decimal($amount);
		}

		$quantity = (float)$this->get('quantity') + (float)$amount;

		$this->set('quantity', $quantity);

		return $quantity;
	}

	/**
	 * @return string
	 */
	public function notes()
	{
		return $this->get('notes');
	}

	/**
	 * @param string $value
	 */
	public function set_notes($value)
	{
		$this->set('notes', trim($value));
	}

	/**
	 * @return int|float|null If view context, will always be a number, and defaults to WooCommerce setting
	 */
	public function low_stock()
	{
		$value = $this->get('low_stock');

		if ($this->context === 'view') {
			if ($value === null) {
				$value = get_option('woocommerce_notify_low_stock_amount', 2);
			}

			$value = (float)$value;
		} elseif ($value !== null) {
			$value = (float)$value;
		}

		return $value;
	}

	/**
	 * @param int|float|null $value
	 */
	public function set_low_stock($value)
	{
		if ($value === '') {
			$value = null;
		} elseif ($value !== null) {
			if (is_string($value)) {
				$value = Number::period_decimal($value);
			}

			$value = (float)$value;
		}

		$this->set('low_stock', $value);
	}

	/**
	 * @return bool
	 */
	public function limit_products()
	{
		return (bool)$this->get('limit_products');
	}

	/**
	 * @param bool
	 */
	public function set_limit_products($value)
	{
		$this->set('limit_products', (bool)$value);
	}

	/**
	 * @return bool
	 */
	public function match_all()
	{
		return (bool)$this->get('match_all');
	}

	/**
	 * @param bool
	 */
	public function set_match_all($value)
	{
		$this->set('match_all', (bool)$value);
	}

	/**
	 * @return array A list of product IDs
	 */
	public function products()
	{
		$value = $this->get('products');

		return $value ? (array)$value : [];
	}

	/**
	 * @param array $value A list of product IDs
	 */
	public function set_products($value)
	{
		$this->set('products', array_map('intval', $value));
	}

	/**
	 * @return array A list of product IDs
	 */
	public function exclude_products()
	{
		$value = $this->get('exclude_products');

		return $value ? (array)$value : [];
	}

	/**
	 * @param array $value A list of product IDs
	 */
	public function set_exclude_products($value)
	{
		$this->set('exclude_products', array_map('intval', $value));
	}

	/**
	 * @return array A list of product category IDs
	 */
	public function categories()
	{
		$value = $this->get('categories');

		return $value ? (array)$value : [];
	}

	/**
	 * @param array $value A list of product category IDs
	 */
	public function set_categories($value)
	{
		$this->set('categories', array_map('intval', $value));
	}

	/**
	 * @return array A list of product category IDs
	 */
	public function exclude_categories()
	{
		$value = $this->get('exclude_categories');

		return $value ? (array)$value : [];
	}

	/**
	 * @param array $value A list of product category IDs
	 */
	public function set_exclude_categories($value)
	{
		$this->set('exclude_categories', array_map('intval', $value));
	}

	/**
	 * @return array
	 */
	public function product_types()
	{
		$value = $this->get('product_types');

		return $value ? (array)$value : [];
	}

	/**
	 * @param array $value
	 */
	public function set_product_types($value)
	{
		$this->set('product_types', $value);
	}

	/**
	 * @inheritDoc
	 */
	public function duplicate($data = [], $context = 'view')
	{
		$copy = parent::duplicate($data, $context);

		if ($copy) {
			$match_sets = Matches::get_sets($this->id, 'edit');
			Matches::save_sets($copy->id(), $match_sets);

			foreach ($this->meta('attribute_level', false) as $attribute_id) {
				$copy->add_meta('attribute_level', $attribute_id);
			}
		}

		do_action(static::hook_name('duplicated_attributes'), $copy, $this, $data);

		return $copy;
	}

	public function formatted_quantity()
	{
		global $wp_locale;

		$quantity = $this->get_contextual('quantity');
		$parts = explode('.', Number::period_decimal($quantity), 2);

		$html = '<span class="qty-int">' . number_format_i18n($parts[0]) . '</span>';

		if (isset($parts[1])) {
			$dp = $wp_locale ? $wp_locale->number_format['decimal_point'] : '.';
			$html .= '<span class="qty-dec">' . $dp . $parts[1] . '</span>';
		}

		$html = apply_filters(static::hook_name('formatted_quantity'), $html, $this);

		$classes = [];

		if ($quantity <= $this->get_contextual('low_stock')) {
			$classes[] = 'low-stock';
		}

		$classes = apply_filters(static::hook_name('formatted_quantity_classes'), $classes, $this);

		return '<span class="stock-quantity' . ($classes ? ' ' . implode(' ', $classes) : '') . '">' . $html . '</span>';
	}
}
