<?php
namespace Mewz\WCAS\Util;

use Mewz\WCAS\Models\AttributeStock;

class Orders
{
	const REDUCED_STOCK_META = '_mewz_wcas_reduced_stock';

	/**
	 * Update attribute stock for all line items of the specified order.
	 *
	 * Note: This will add an order note to the order.
	 *
	 * @param \WC_Order|int $order
	 * @param string $operation 'reduce', 'restore' or 'sync'
	 *
	 * @return bool
	 */
	public static function update_order_attribute_stock($order, $operation = 'reduce')
	{
		if (!in_array($operation, ['reduce', 'restore', 'sync'])) {
			return false;
		}

		if (!$order instanceof \WC_Order && !$order = wc_get_order($order)) {
			return false;
		}

		return self::update_order_items_attribute_stock($order->get_items(), $operation, $order);
	}

	/**
	 * Update attribute stock for one or more order line items.
	 *
	 * Important: All order items must be from the same order else hellfire shall rain from the skies.
	 *
	 * Note: This will add an order note to the order.
	 *
	 * @param \WC_Order_Item_Product|\WC_Order_Item_Product[]|int|array $order_items
	 * @param string $operation 'reduce', 'restore' or 'sync'
	 * @param \WC_Order|int $order
	 *
	 * @return array|bool
	 */
	public static function update_order_items_attribute_stock($order_items, $operation = 'reduce', $order = null)
	{
		if (!$order_items || !in_array($operation, ['reduce', 'restore', 'sync'])) {
			return false;
		}

		if ($order !== null && !$order instanceof \WC_Order) {
			$order = wc_get_order($order);
		}

		$changes = [];

		foreach ((array)$order_items as $order_item) {
			if (!$order_item) continue;

			if (is_array($order_item)) {
				if (empty($order_item[0])) continue;

				$quantity = isset($order_item[1]) ? $order_item[1] : null;
				$order_item = $order_item[0];
			} else {
				$quantity = null;
			}

			$order_item = self::get_valid_order_line_item($order_item);
			if (!$order_item) continue;

			$change = self::update_order_item_attribute_stock($order_item, $operation, $quantity);
			if ($change === false) continue;

			$changes[] = $change;

			if (!$order) {
				$order = $order_item->get_order();
			}
		}

		if ($changes) {
			$changes = self::merge_changes(...$changes);

			if ($changes) {
				self::add_stock_change_order_note($order, $operation, $changes);
			}
		}

		if ($order) {
			do_action('mewz_wcas_order_attribute_stock_updated', $order, $operation, $changes);
		}

		return $changes;
	}

	/**
	 * Update attribute stock for a single order line item.
	 *
	 * Note: This will NOT add an order note to the order.
	 *
	 * @param \WC_Order_Item_Product|int $order_item
	 * @param string $operation 'reduce', 'restore' or 'sync'
	 * @param float $quantity If set, stock is updated by this amount regardless of whether it
	 *                        has been updated before. Useful for incremental stock updates.
	 *
	 * @return array|false
	 */
	public static function update_order_item_attribute_stock($order_item, $operation = 'reduce', $quantity = null)
	{
		if (!in_array($operation, ['reduce', 'restore', 'sync'])) {
			return false;
		}

		if (!$order_item = self::get_valid_order_line_item($order_item)) {
			return false;
		}

		$product = $order_item->get_product() ?: new \WC_Product_Simple();

		if (Products::is_product_excluded($product)) {
			return false;
		}

		$reduced_qty = wc_stock_amount(max((float)$order_item->get_meta(self::REDUCED_STOCK_META), 0));

		if ($operation === 'sync') {
			$item_qty = self::get_effective_order_item_quantity($order_item);

			if ($item_qty > $reduced_qty) {
				$operation = 'reduce';
				$quantity = $item_qty - $reduced_qty;
			} elseif ($reduced_qty > $item_qty) {
				$operation = 'restore';
				$quantity = $reduced_qty - $item_qty;
			} else {
				return [];
			}
		}

		$reduce = $operation === 'reduce';
		$update_allowed = $quantity !== null || $reduce !== (bool)$reduced_qty;

		$update_allowed = apply_filters('mewz_wcas_update_order_item_attribute_stock', $update_allowed, $order_item, $product, $operation, $quantity, $reduced_qty);
		if (!$update_allowed) return false;

		$attributes = self::get_order_item_attributes($order_item, $product);
		$matches = Matches::match_product_stock($product, $attributes);

		$matches = apply_filters('mewz_wcas_order_item_stock_matches', $matches, $order_item, $attributes, $product, $operation, $quantity);
		if (!$matches) return false;

		if ($quantity === null) {
			$quantity = $reduce ? self::get_effective_order_item_quantity($order_item) : $reduced_qty;
		}

		$quantity = apply_filters('mewz_wcas_update_order_item_quantity', $quantity, $matches, $order_item, $attributes, $product, $operation);

		if (!$quantity) return false;

		$changes = [];
		$reduce_sign = $reduce ? -1 : 1;

		foreach ($matches as $match) {
			$adjust_amount = $quantity * $match['multiplier'];
			$adjust_amount = apply_filters('mewz_wcas_update_order_item_attribute_stock_amount', $adjust_amount, $match, $matches, $order_item, $attributes, $product, $operation, $quantity);

			if (!$adjust_amount) continue;

			$adjust_amount *= $reduce_sign;

			$stock = AttributeStock::instance($match['stock_id'], 'edit');

			$prev_quantity = $stock->quantity();
			$new_quantity = $stock->adjust_quantity($adjust_amount);

			$stock->save();

			$changes[$stock->id()] = [
				'amount' => $adjust_amount,
				'from' => $prev_quantity,
				'to' => $new_quantity,
			];

			if ($reduce) {
				do_action('mewz_wcas_trigger_stock_notification', $stock, $prev_quantity);
			}
		}

		$reduced_qty += $reduce ? $quantity : -$quantity;

		if ($reduced_qty > 0) {
			$order_item->update_meta_data(self::REDUCED_STOCK_META, $reduced_qty);
		} else {
			$order_item->delete_meta_data(self::REDUCED_STOCK_META);
		}

		$order_item->save_meta_data();

		do_action('mewz_wcas_order_item_attribute_stock_updated', $order_item, $operation, $reduced_qty, $changes);

		return $changes;
	}

	/**
	 * @param \WC_Order_Item|int $order_item
	 *
	 * @return \WC_Order_Item_Product|false
	 */
	public static function get_valid_order_line_item($order_item)
	{
		if (!$order_item instanceof \WC_Order_Item && !$order_item = \WC_Order_Factory::get_order_item($order_item)) {
			return false;
		}

		if (!$order_item->get_id() || $order_item->get_type() !== 'line_item') {
			return false;
		}

		return $order_item;
	}

	/**
	 * @param \WC_Order_Item_Product $order_item
	 * @param \WC_Product $product
	 *
	 * @return array|false
	 */
	public static function get_order_item_attributes(\WC_Order_Item_Product $order_item, \WC_Product $product = null)
	{
		if ($product === null) {
			$product = $order_item->get_product();
		}

		// get product attributes
		$attributes = $product ? Products::get_product_attributes($product, false, true) : [];

		// merge order item attributes
		foreach ($order_item->get_meta_data() as $meta) {
			if (strpos($key = $meta->key, 'pa_') === 0) {
				$attributes[$key] = $meta->value;
			}
		}

		return apply_filters('mewz_wcas_order_item_attributes', $attributes, $order_item, $product);
	}

	/**
	 * @param \WC_Order_Item $order_item
	 *
	 * @return float
	 */
	public static function get_effective_order_item_quantity(\WC_Order_Item $order_item)
	{
		$refunded_qty = $order_item->get_order()->get_qty_refunded_for_item($order_item->get_id()); // returns a negative number

		return wc_stock_amount($order_item->get_quantity() + $refunded_qty);
	}

	/**
	 * @param array ...$changes
	 *
	 * @return array
	 */
	public static function merge_changes(...$changes)
	{
		$merged = [];

		foreach ($changes as $change) {
			if (!$change) continue;

			foreach ($change as $stock_id => $values) {
				if (!isset($merged[$stock_id])) {
					$merged[$stock_id] = $values;
				} else {
					$merged[$stock_id]['amount'] += $values['amount'];
					$merged[$stock_id]['to'] = $values['to'];
				}
			}
		}

		return $merged;
	}

	/**
	 * @param \WC_Order|int $order
	 * @param string $operation
	 * @param array $changes
	 */
	public static function add_stock_change_order_note($order, $operation, array $changes)
	{
		if (!$order instanceof \WC_Order) {
			$order = wc_get_order($order);
		}

		if (!$order) return;

		$changes = apply_filters('mewz_wcas_order_note_stock_changes', $changes, $order, $operation);

		if (!$changes) return;

		$lines = [];

		foreach ($changes as $stock_id => $change) {
			if (!isset($change['from'], $change['to']) || $change['from'] == $change['to']) {
				continue;
			}

		    $stock = AttributeStock::instance($stock_id);

		    $sku = $stock->sku();
		    $title = $stock->title() . ($sku != '' ? " ($sku)" : '');

			$from = apply_filters(AttributeStock::hook_name('get_quantity'), $change['from'], $stock);
			$to = apply_filters(AttributeStock::hook_name('get_quantity'), $change['to'], $stock);

			$lines[] = sprintf('%s %s &rarr; %s', $title, $from, $to);
		}

		if (!$lines) return;

		if ($operation === 'reduce') {
			$title = __('Attribute stock reduced', 'woocommerce-attribute-stock');
		} elseif ($operation === 'restore') {
			$title = __('Attribute stock increased', 'woocommerce-attribute-stock');
		} else {
			$title = __('Attribute stock adjusted', 'woocommerce-attribute-stock');
		}

		$order_note = $title . ":\n" . implode("\n", $lines);
		$order_note = apply_filters('mewz_wcas_order_note', $order_note, $order, $operation, $changes, $title, $lines);

		if ($order_note) {
			$order->add_order_note($order_note);
		}
	}
}
