<?php

namespace TenWebIO;

use TenWebQueue\DTO\QueueConfigDTO;
use TenWebQueue\DTO\QueueDataDTO;
use TenWebQueue\QueueOrchestrator;
use TenWebQueue\Exceptions\QueueException;
use \TenWebQueue\QueueContext;

class CompressService
{
    const QUEUE_NAME              = 'tenweb_image_optimizer';
    const QUEUE_RESTART_FILE_NAME = 'tenweb_image_optimizer_restart';
    /**
     * @var Settings
     */
    private $compress_settings;

    /**
     * @var Config
     */
    private $config;

    /**
     * @var int
     */
    private $restart = 0;

    /**
     * @var string
     */
    private $restart_route;


    /**
     * @var array
     */
    private $restart_body;

    /**
     * @var array
     */
    private $images = array();

    /**
     * @param $restart
     */
    public function __construct($restart = 0)
    {
        $this->compress_settings = new Settings();
        $this->config = new Config();
        $this->restart = $restart;
    }

    /**
     *
     * @return void
     * @throws QueueException
     */
    public function compressBulk()
    {
        if (!$this->restart) {
            $this->images = $this->getImagesReadyForOptimize(true)->getData();
            if (empty($this->images)) {
                return;
            }
        }
        $this->restart_route = add_query_arg(array('rest_route' => '/tenwebio/v2/compress'), get_home_url() . "/");
        $this->compress('bulk');
    }

    /**
     *
     * @param     $image_urls
     * @param     $page_id
     * @param int $only_convert_webp
     *
     * @return void
     * @throws QueueException
     */
    public function compressCustom($image_urls, $page_id, $only_convert_webp = 0)
    {
        if (!$this->restart) {
            $attachments = new Attachments();
            $this->images = $attachments->getCustomAttachmentsData($image_urls);
            if (empty($this->images)) {
                return;
            }
        }
        $this->restart_route = add_query_arg(array('rest_route' => '/tenwebio/v2/compress-custom'), get_home_url() . "/");
        $this->restart_body['page_id'] = $page_id;
        $this->restart_body['only_convert_webp'] = $only_convert_webp;
        set_transient(TENWEBIO_PREFIX . '_custom_compress_only_convert_webp_' . $page_id, $only_convert_webp);
        $this->compress('custom_' . $page_id);
    }

    /**
     * @param $post_id
     *
     * @return void
     * @throws QueueException
     */
    public function compressOne($post_id)
    {
        if (!$this->restart) {
            $this->images = $this->getImageReadyForOptimize($post_id)->getData();
            if (empty($this->images)) {
                return;
            }
        }
        $this->restart_route = add_query_arg(array('rest_route' => '/tenwebio/v2/compress-one'), get_home_url() . "/");
        $this->compress('single_' . $post_id);
    }

    /**
     * @param     $type
     *
     * @return void
     * @throws QueueException
     */
    public function compress($type)
    {
        Logs::setLog("compress:" . $type . ":restart", $this->restart);
        $queue = $this->createQueue($type);

        if (!$this->restart) {
            Utils::deleteQueueTransients($type, false);
            set_transient(TENWEBIO_PREFIX . '_compress_images_count_' . $type, count($this->images), 3600 * 4);
            $queue->enqueue($this->images);
        }
        $queue->dequeue();
    }

    /***
     * @param $type
     *
     * @return QueueContext
     * @throws QueueException
     */
    public function createQueue($type)
    {
        $queue_dir = Utils::getQueueDir($type);
        $this->restart_body['restart'] = 1;
        $this->restart_body['tenwebio_nonce'] = wp_create_nonce('tenwebio_rest');

        return QueueOrchestrator::initQueue(new QueueConfigDTO([
            'connection_type'         => 'file',
            'file_connection_path'    => $queue_dir,
            'restart_file_path'       => $queue_dir . '/' . self::QUEUE_RESTART_FILE_NAME . '_' . $type . '.txt',
            'restart_route'           => $this->restart_route,
            'restart_body'            => $this->restart_body,
            'items_count_for_restart' => $this->config->getImagesLimitForRestart(),
            'force'                   => (!$this->restart),
            'queue_data'              => new QueueDataDTO([
                'name'           => self::QUEUE_NAME . '_' . $type,
                'multiple_items' => true
            ])
        ]), \TenWebIO\Queue\QueueProducer::class, \TenWebIO\Queue\QueueConsumer::class);
    }

    /**
     * @param bool $force_stat
     *
     * @return array
     */
    public function getLog($force_stat = false)
    {
        $not_optimized = $this->getImagesReadyForOptimize($force_stat)->getDataSeparate();
        $last_optimized = new LastCompress();
        $size = $last_optimized->getImageOrigSize() - $last_optimized->getImageSize();
        $percent = $last_optimized->getImageOrigSize() ? ($size / $last_optimized->getImageOrigSize()) * 100 : 0;

        return array(
            'not_optimized'  => array(
                'full'       => count($not_optimized['attachments_full']),
                'thumbs'     => count($not_optimized['attachments_meta']),
                'other'      => count($not_optimized['attachments_other']),
                'total_size' => $not_optimized['total_size']
            ),
            'last_optimized' => array(
                'size'    => $size,
                'percent' => $percent
            )
        );
    }


    /**
     * @param $force_stat
     * @param $first_in_queue
     *
     * @return Attachments
     */
    public function getImagesReadyForOptimize($force_stat = false, $first_in_queue = array())
    {
        $stat = $this->compress_settings->getStat($force_stat, 0, 1);
        $settings = $this->compress_settings->getSettings(false, 1, 1);

        $exclude_ids = !empty($stat['full_ids']) ? $stat['full_ids'] : array();
        $exclude_thumb_ids = !empty($stat['thumb_ids']) ? $stat['thumb_ids'] : array();
        $exclude_other = !empty($stat['other']) ? $stat['other'] : array();
        $other_directories = !empty($settings['other_directories']) ? $settings['other_directories'] : array();

        $attachments = new Attachments();
        $attachments->setExcludedIds($exclude_ids);
        $attachments->setExcludedThumbIds($exclude_thumb_ids);
        $attachments->setOtherDirectories($other_directories);
        $attachments->setExcludedOtherPaths($exclude_other);
        $attachments->setFirstInQueue($first_in_queue);

        return $attachments;
    }

    /**
     * @param $post_id
     * @param $force_stat
     *
     * @return Attachments
     */
    public function getImageReadyForOptimize($post_id, $force_stat = false)
    {
        $stat = $this->compress_settings->getStat($force_stat, 1, 1);
        $exclude_ids = !empty($stat['full_ids']) ? $stat['full_ids'] : array();
        $exclude_thumb_ids = !empty($stat['thumb_ids']) ? $stat['thumb_ids'] : array();

        $attachments = new Attachments();
        $attachments->setExcludedIds($exclude_ids);
        $attachments->setExcludedThumbIds($exclude_thumb_ids);
        $attachments->setFilteredIds(array($post_id));

        return $attachments;
    }

    /**
     * @param      $data
     * @param bool $is_finished
     *
     * @return void
     */
    public function storeDataInService($data, $is_finished = 0)
    {
        $api_instance = new Api(Api::API_COMPRESS_LOG_STORE_ACTION);
        $api_instance->apiRequest('POST', array('data' => $data, 'is_finished' => $is_finished));
    }

    /**
     * Counter can be number, or other directory name like elementor
     *
     * @param string  $counter
     *
     * @param integer $webP_paths
     *
     * @return array
     */
    public function getAlreadyConvertedImagesPaths($counter = '1', $webP_paths = 1)
    {
        $data = get_option(TENWEBIO_PREFIX . '_converted_images_' . $counter);
        if (empty($data)) {
            return array();
        }
        $data = json_decode($data);
        $paths = array();
        foreach ($data as $row) {
            $row = json_decode($row, true);
            $paths[] = $webP_paths ? $row['path'] : rtrim($row['path'], '.webp');
        }

        return array_unique($paths);
    }

}
