<?php
namespace Mewz\WCAS\Util;

use Mewz\WCAS\Actions\Front\ProductLimits;

class Cart
{
	public static $cache = [];

	public static function get_cart_validation_items()
	{
		return apply_filters('mewz_wcas_cart_validation_items', WC()->cart->get_cart());
	}

	public static function get_cart_attribute_stock_limits($adding_key = null)
	{
		$limits = [];
		$cart_items = self::get_cart_validation_items();

		foreach ($cart_items as $item_key => $item) {
			if (!Products::is_product_allowed($item['data'])) {
				continue;
			}

			$variation = !empty($item['variation']) ? $item['variation'] : [];

			$item_limits = Limits::get_stock_limits($item['data'], $variation);
			if (!$item_limits) continue;

			foreach ($item_limits as $stock_id => $limit) {
				$cart_quantity = $limit['multiplier'] * $item['quantity'];
				$cart_quantity = (int)apply_filters('mewz_wcas_cart_stock_item_quantity', $cart_quantity, $item, $limit, $item_limits, $adding_key);

				if (!isset($limits[$stock_id]['stock_qty'])) {
					$limits[$stock_id]['stock_qty'] = $limit['stock_qty'];
				}

				if (!isset($limits[$stock_id]['cart_qty'])) {
					$limits[$stock_id]['cart_qty'] = 0;
				}

				$limits[$stock_id]['cart_qty'] += $cart_quantity;
				$limits[$stock_id]['cart_items'][$item_key] = $item;

				/** @deprecated */
				$limits[$stock_id]['cart_limits'][$item_key] = $limit;
			}
		}

		return apply_filters('mewz_wcas_cart_stock_limits', $limits, $cart_items, $adding_key);
	}

	public static function get_adding_over_shared_attribute_stock(\WC_Product $product, $quantity = 1, $variation = null, $cart_item_key = null)
	{
		if (!Products::is_product_allowed($product)) {
			return false;
		}

		$quantity = apply_filters('mewz_wcas_cart_adding_over_shared_quantity', $quantity, $product, $variation, $cart_item_key);

		if ($quantity <= 0) {
			return false;
		}

		$adding_limits = Limits::get_stock_limits($product, $variation);
		$adding_limits = apply_filters('mewz_wcas_cart_adding_over_shared_adding_limits', $adding_limits, $product, $variation, $quantity, $cart_item_key);
		if (!$adding_limits) return false;

		$cart_limits = self::get_cart_attribute_stock_limits($cart_item_key);
		$cart_limits = apply_filters('mewz_wcas_cart_adding_over_shared_cart_limits', $cart_limits, $adding_limits, $product, $variation, $quantity, $cart_item_key);
		if (!$cart_limits) return false;

		$over_limits = [];

		foreach ($adding_limits as $stock_id => $adding_limit) {
			if (!isset($cart_limits[$stock_id])) continue;

			$cart_limit = $cart_limits[$stock_id];

			$adding_qty = $adding_limit['multiplier'] * $quantity;
			$adding_qty = apply_filters('mewz_wcas_cart_adding_over_shared_adding_quantity', $adding_qty, $adding_limit, $cart_limit, $product, $variation, $quantity, $cart_item_key);

			if ($adding_qty <= 0) {
				continue;
			}

			if ($cart_limit['cart_qty'] + $adding_qty > $adding_limit['stock_qty']) {
				$adding_limit['adding_qty'] = $adding_qty;
				$cart_limit['adding_limit'] = $adding_limit;
				$over_limits[$stock_id] = $cart_limit;
			}
		}

		return $over_limits;
	}

	public static function get_over_shared_attribute_stock()
	{
		if (!isset(self::$cache['over_limits'])) {
			$cart_limits = self::get_cart_attribute_stock_limits();
			if (!$cart_limits) return false;

			$over_limits = [];

			foreach ($cart_limits as $stock_id => $limit) {
				if ($limit['cart_qty'] > $limit['stock_qty']) {
					$over_limits[$stock_id] = $limit;
				}
			}

			self::$cache['over_limits'] = apply_filters('mewz_wcas_cart_over_shared_stock_limits', $over_limits, $cart_limits);
		}

		return self::$cache['over_limits'];
	}

	/**
	 * Set selected variation data on cart item variation product objects so that correct
	 * attribute stock can be calculated for variations with catch-all attributes.
	 *
	 * @param array $cart_item
	 *
	 * @return false|mixed
	 */
	public static function set_item_product_variation_data($cart_item)
	{
		if (empty($cart_item['data'])) {
			return false;
		}

		/** @var \WC_Product $product */
		$product = $cart_item['data'];

		if (!empty($cart_item['variation_id']) && !empty($cart_item['variation']) && is_array($cart_item['variation']) && $product instanceof \WC_Product_Variation && Attributes::has_catchall($product->get_attributes())) {
			$product->mewz_wcas_variation = $cart_item['variation'];
			return $cart_item['variation'];
		} elseif (property_exists($product, 'mewz_wcas_variation')) {
			unset($product->mewz_wcas_variation);
		}

		return false;
	}

	/**
	 * Check that product being added to cart is in stock and has enough stock (not accounting for quantity already in the cart).
	 *
	 * Error messages are the default WooCommerce messages from {@see WC_Cart::add_to_cart()}.
	 *
	 * @param \WC_Product $product
	 * @param float $quantity
	 *
	 * @return true|\WP_Error
	 */
	public static function validate_adding_stock(\WC_Product $product, $quantity)
	{
		if (!$product->is_in_stock()) {
			$message = sprintf(__('You cannot add &quot;%s&quot; to the cart because the product is out of stock.', 'woocommerce'), $product->get_name());

			$message = apply_filters('woocommerce_cart_product_out_of_stock_message', $message, $product);
		}

		if (!$product->has_enough_stock($quantity)) {
			$message = sprintf(__('You cannot add that amount of &quot;%1$s&quot; to the cart because there is not enough stock (%2$s remaining).', 'woocommerce'), $product->get_name(), wc_format_stock_quantity_for_display($product->get_stock_quantity(), $product));

			$message = apply_filters('woocommerce_cart_product_not_enough_stock_message', $message, $product, $quantity);
		}

		if (isset($message)) {
			return new \WP_Error('insufficient_stock', $message);
		} else {
			return true;
		}
	}

	/**
	 * Check that variation being added to cart has enough product-level stock (accounting for quantity already in the cart).
	 *
	 * Stock limited variations are filtered to have 'manage_stock' = true even if using product-level stock ('manage_stock' = parent)
	 * in order to avoid issues with parent stock validations when adding to cart. Specifically, parent stock is retrieved with `$variation->get_stock_quantity()`
	 * which, when variation stock is limited by attribute stock, does not reliably get the correct quantity of the product-level stock.
	 *
	 * Since native stock checking of variation product-level stock is now essentially disabled for stock limited variations,
	 * we need to perform the checks ourselves.
	 *
	 * Error message is the default WooCommerce message from {@see WC_Cart::add_to_cart()}.
	 *
	 * @param \WC_Product_Variation $product
	 * @param float $quantity
	 *
	 * @return true|\WP_Error
	 */
	public static function validate_adding_parent_stock(\WC_Product_Variation $product, $quantity)
	{
		// all attribute stock limits should be bypassed as we only want to validate actual product stock
		ProductLimits::$enabled = false;

		if ($product->managing_stock() === 'parent' && $parent_id = $product->get_parent_id()) {
			$parent_cart_qty = 0;

			foreach (self::get_cart_validation_items() as $item) {
				if ($item['data']->get_stock_managed_by_id() === $parent_id) {
					$parent_cart_qty += $item['quantity'];
				}
			}

			if ($parent_cart_qty > 0 && ($parent_stock_qty = $product->get_stock_quantity()) < $parent_cart_qty + $quantity) {
				$message = sprintf(__('You cannot add that amount to the cart &mdash; we have %1$s in stock and you already have %2$s in your cart.', 'woocommerce'), wc_format_stock_quantity_for_display($parent_stock_qty, $product), wc_format_stock_quantity_for_display($parent_cart_qty, $product));

				$message = sprintf('<a href="%s" class="button wc-forward">%s</a> %s', esc_url(wc_get_cart_url()), esc_html__('View cart', 'woocommerce'), esc_html($message));

				$message = apply_filters('woocommerce_cart_product_not_enough_stock_already_in_cart_message', $message, $product, $parent_stock_qty, $parent_cart_qty);
			}
		}

		ProductLimits::$enabled = true;

		return isset($message) ? new \WP_Error('insufficient_parent_stock', $message) : true;
	}

	/**
	 * Check that there is enough (shared) attribute stock for product being added to cart and products in the cart.
	 *
	 * @param \WC_Product $product
	 * @param float $quantity
	 *
	 * @return true|\WP_Error
	 */
	public static function validate_adding_shared_attribute_stock(\WC_Product $product, $quantity, $cart_item_key)
	{
		$over_limits = self::get_adding_over_shared_attribute_stock($product, $quantity, null, $cart_item_key);
		if (!$over_limits) return true;

		$limit = current($over_limits);
		$limit = apply_filters('mewz_wcas_cart_add_stock_error_limit', $limit, $product, $quantity, $over_limits, $cart_item_key);

		$message = sprintf(__('You cannot add that amount to the cart &mdash; we have %1$s in stock and you already have %2$s in your cart.', 'woocommerce'), wc_format_stock_quantity_for_display($limit['stock_qty'], $product), wc_format_stock_quantity_for_display($limit['cart_qty'], $product));

		$message = apply_filters('mewz_wcas_cart_add_stock_error', $message, $product, $quantity, $over_limits, $cart_item_key);

		$message = sprintf('<a href="%s" class="button wc-forward">%s</a> %s', esc_url(wc_get_cart_url()), esc_html__('View cart', 'woocommerce'), esc_html($message));

		$message = apply_filters('woocommerce_cart_product_not_enough_stock_already_in_cart_message', $message, $product, $limit['stock_qty'], $limit['cart_qty']);

		return new \WP_Error('insufficient_attribute_stock', $message);
	}

	/**
	 * Modified version of {@see \WC_Cart::check_cart_item_stock()} to check stock
	 * of cart items, but without taking into account stock of other items in the cart
	 * (we need to do this separately in a way that works with attribute stock).
	 *
	 * @return true|\WP_Error
	 */
	public static function check_individual_items_stock()
	{
		$current_session_order_id = isset(WC()->session->order_awaiting_payment) ? absint(WC()->session->order_awaiting_payment) : 0;

		foreach (WC()->cart->get_cart_contents() as $item) {
			/** @var \WC_Product $product */
			$product = $item['data'];

			if (!$product->is_in_stock()) {
				return new \WP_Error('out-of-stock', sprintf(__('Sorry, "%s" is not in stock. Please edit your cart and try again. We apologize for any inconvenience caused.', 'woocommerce'), $product->get_name()));
			}

			if (!$product->managing_stock() || $product->backorders_allowed()) {
				continue;
			}

			$held_stock = wc_get_held_stock_quantity($product, $current_session_order_id);

			/**
			 * Allows filter if product have enough stock to get added to the cart.
			 *
			 * @param bool $has_stock If have enough stock.
			 * @param \WC_Product $product Product instance.
			 * @param array $item Cart item values.
			 *
			 * @since 4.6.0
			 */
			if (apply_filters('woocommerce_cart_item_required_stock_is_not_enough', $product->get_stock_quantity() < ($held_stock + $item['quantity']), $product, $item)) {
				return new \WP_Error('out-of-stock', sprintf(__('Sorry, we do not have enough "%1$s" in stock to fulfill your order (%2$s available). We apologize for any inconvenience caused.', 'woocommerce'), $product->get_name(), wc_format_stock_quantity_for_display($product->get_stock_quantity() - $held_stock, $product)));
			}
		}

		return true;
	}

	/**
	 * Check that there is enough (shared) attribute stock for all items currently in the cart.
	 *
	 * @return true|\WP_Error
	 */
	public static function check_shared_attribute_stock()
	{
		$over_limits = self::get_over_shared_attribute_stock();
		if (!$over_limits) return true;

		foreach ($over_limits as $limit) {
			if (count($limit['cart_items']) <= 1) {
				continue;
			}

			// we need backorder items to calculate stock limits for non-backorder items
			// but if all cart items are backorder items, then over-purchasing is allowed
			$all_allow_backorders = true;

			foreach ($limit['cart_items'] as $cart_item) {
				if (!$cart_item['data']->backorders_allowed()) {
					$all_allow_backorders = false;
				}
			}

			if ($all_allow_backorders) continue;

			// we have stock being over-purchased, so show an error
			$product_names = [];

			foreach ($limit['cart_items'] as $cart_item) {
				$product_names[] = $cart_item['data']->get_name();
			}

			$product_names = implode(', ', array_unique($product_names));

			$message = sprintf(__('Sorry, we do not have enough "%1$s" in stock to fulfill your order (%2$s available). We apologize for any inconvenience caused.', 'woocommerce'), $product_names, wc_format_stock_quantity_for_display($limit['stock_qty'], current($limit['cart_items'])['data']));

			$message = apply_filters('mewz_wcas_cart_stock_error', $message, $limit, $over_limits);

			return new \WP_Error('insufficient_attribute_stock', $message);
		}

		return true;
	}
}
