<?php
namespace Mewz\WCAS\Util;

use Mewz\Framework\Util\WooCommerce;
use Mewz\WCAS\Actions\Front\ProductLimits;
use Mewz\WCAS\Models\AttributeStock;

class Limits
{
	/**
	 * Checks if product limits are loaded and active
	 *
	 * @return bool
	 */
	public static function product_limits_active()
	{
		static $loaded = false;

		if (!$loaded) {
			$loaded = class_exists(ProductLimits::class, false);
		}

		return $loaded && ProductLimits::$enabled;
	}

	/**
	 * Retrieves a list of matching attribute stock quantity data for a product or variation,
	 * sorted from lowest to highest stock.
	 *
	 * This is the main method used to determine a product's available stock on the frontend
	 * based on attribute stock items with the "Limit product stock" setting enabled.
	 *
	 * @param \WC_Product|int $product Product object or ID
	 * @param array $variation Selected variation key/value pairs
	 *
	 * @return array|false
	 */
	public static function get_stock_limits($product, $variation = null)
	{
		if (!$product instanceof \WC_Product && !$product = wc_get_product($product)) {
			return false;
		}

		if ($variation !== null && !is_array($variation)) {
			$variation = [];
		}

		if ($variation === null && property_exists($product, 'mewz_wcas_variation') && $product->mewz_wcas_variation) {
			$variation = $product->mewz_wcas_variation;
		}

		$cache_key = 'product_stock_limits';
		$cache_tags = ['match_sets', 'stock', 'product_' . $product->get_id()];

		if ($variation) {
			ksort($variation);
			$cache_key .= '_' . md5(json_encode($variation));
		}

		$limits = Mewz_WCAS()->cache->get($cache_key, $cache_tags);

		if (!is_array($limits)) {
			$limits = [];

			$attributes = Products::get_product_attributes($product, $variation);
			$matches = Matches::match_product_stock($product, $attributes);

			if ($matches) {
				foreach ($matches as $match) {
					$stock = AttributeStock::instance($match['stock_id']);

					if ($stock->limit_products()) {
						$stock_qty = $stock->quantity();
						$limit_qty = Matches::get_limit_qty($stock_qty, $match['multiplier']);

						$limits[$match['stock_id']] = $match + compact('stock_qty', 'limit_qty');
					}
				}

				if (count($limits) > 1) {
					self::sort_limits($limits);
				}
			} elseif (
				$variation
				&& $product instanceof \WC_Product_Variation
				&& Settings::unmatched_any_variations(true) !== 'no'
				&& Attributes::has_catchall($product->get_attributes())
			) {
				$limits[0] = [
					'stock_id' => 0,
					'multiplier' => 1,
					'stock_qty' => 0,
					'limit_qty' => 0,
				];
			}

			/**
			 * IMPORTANT: The results from this filter are cached for performance reasons.
			 * This means the input ($product, $variation) should always have the same output,
			 * or the cache must be invalidated accordingly.
			 */
			$limits = apply_filters('mewz_wcas_product_stock_limits', $limits, $product, $attributes, $variation, $matches);

			Mewz_WCAS()->cache->set($cache_key, $limits, $cache_tags);
		}

		$limits = apply_filters('mewz_wcas_product_stock_limits_uncached', $limits, $product, $variation);

		return $limits;
	}

	public static function sort_limits(&$limits)
	{
	    return uasort($limits, [self::class, 'sort_by_limit_qty']);
	}

	protected static function sort_by_limit_qty($a, $b)
	{
		return $a['limit_qty'] - $b['limit_qty'];
	}

	/**
	 * @param \WC_Product|int $product
	 * @param array $variation
	 *
	 * @return array|mixed
	 */
	public static function get_stock_limit($product, $variation = null)
	{
		$limits = self::get_stock_limits($product, $variation);
		if (!$limits) return false;

		if (property_exists($product, 'mewz_wcas_match_highest_limit') && $product->mewz_wcas_match_highest_limit) {
			return end($limits);
		} else {
			return current($limits);
		}
	}

	/**
	 * Gets all 9 variants of stock html messages for a (variable) product.
	 *
	 * @param \WC_Product $product
	 *
	 * @return array
	 */
	public static function get_variable_stock_html(\WC_Product $product)
	{
		// get a new disposable product object so its props can be set freely
		$product = wc_get_product($product->get_id());
		$product->mewz_wcas_bypass_limits = true;
		$product->mewz_wcas_ignore_csr = true;

		return [
			'in' => [
				'no' => self::get_stock_html($product, 'instock', 'no'),
				'yes' => self::get_stock_html($product, 'instock', 'yes'),
				'notify' => self::get_stock_html($product, 'instock', 'notify'),
			],
			'low' => [
				'no' => self::get_stock_html($product, 'lowstock', 'no'),
				'yes' => self::get_stock_html($product, 'lowstock', 'yes'),
				'notify' => self::get_stock_html($product, 'lowstock', 'notify'),
			],
			'out' => [
				'no' => self::get_stock_html($product, 'nostock', 'no'),
				'yes' => self::get_stock_html($product, 'nostock', 'yes'),
				'notify' => self::get_stock_html($product, 'nostock', 'notify'),
			],
		];
	}

	/**
	 * Builds a template string for a particular product stock html message.
	 *
	 * WARNING: This sets props on the product object. Only pass in disposable objects!
	 *
	 * @param \WC_Product $product
	 * @param string $status 'instock', 'lowstock' or 'nostock'
	 * @param string $backorders 'yes', 'no', or 'notify'
	 *
	 * @return string
	 */
	public static function get_stock_html(\WC_Product $product, $status = 'instock', $backorders = 'no')
	{
		static $instock_qty;

		if ($status === 'instock') {
			if ($instock_qty === null) {
				$instock_qty = min(999999999999999, PHP_INT_MAX);
			}

			$quantity = $instock_qty;
		} elseif ($status === 'lowstock') {
			$quantity = WooCommerce::low_stock_amount();
		} elseif ($status === 'nostock') {
			$quantity = 0;
		} else {
			return false;
		}

		$product->set_manage_stock(true);
		$product->set_stock_quantity($quantity);
		$product->set_backorders($backorders);
		$product->validate_props(); // sets 'stock_status'

		$display_qty = wc_format_stock_quantity_for_display($product->get_stock_quantity(), $product);

		$html = wc_get_stock_html($product);
		$html = str_replace($display_qty, '%d', $html);

		return $html;
	}
}
