<?php
namespace Mewz\WCAS\Aspects\Front;

use Mewz\Framework\Base\Aspect;
use Mewz\Framework\Util\WooCommerce;
use Mewz\WCAS\Util\Limits;
use Mewz\WCAS\Util\Matches;
use Mewz\WCAS\Util\Products;
use Mewz\WCAS\Util\Settings;

class ProductLimits extends Aspect
{
	public static $enabled = true;

	public function __hooks()
	{
		add_filter('woocommerce_product_get_manage_stock', [$this, 'get_manage_stock'], 5, 2);
		add_filter('woocommerce_product_get_stock_quantity', [$this, 'get_stock_quantity'], 5, 2);
		add_filter('woocommerce_product_get_stock_status', [$this, 'get_stock_status'], 5, 2);
		add_filter('woocommerce_product_get_image_id', [$this, 'get_image_id'], 5, 2);

		add_filter('woocommerce_product_variation_get_manage_stock', [$this, 'get_manage_stock'], 5, 2);
		add_filter('woocommerce_product_variation_get_stock_quantity', [$this, 'get_stock_quantity'], 5, 2);
		add_filter('woocommerce_product_variation_get_stock_status', [$this, 'get_stock_status'], 5, 2);
		add_filter('woocommerce_product_variation_get_image_id', [$this, 'get_image_id'], 5, 2);
		add_filter('woocommerce_product_variation_get_backorders', [$this, 'variation_get_backorders'], 5, 2);

		add_filter('woocommerce_order_item_product', [$this, 'order_item_product'], 0, 2);

		if (WooCommerce::hide_out_of_stock()) {
			add_filter('woocommerce_get_children', [$this, 'get_visible_children'], 5, 3);
		}

		if (Settings::allow_backorders() !== 'no') {
			add_filter('woocommerce_product_get_backorders', [$this, 'allow_backorders'], 6, 2);
			add_filter('woocommerce_product_variation_get_backorders', [$this, 'allow_backorders'], 6, 2);
		}
	}

	public static function limit_product(\WC_Product $product, $limit_variable = false)
	{
		return self::$enabled && Products::get_prop($product, 'bypass_limits') <= 0 && Products::is_valid_product($product, $limit_variable);
	}

	public static function limit_stock_status(\WC_Product $product)
	{
		return apply_filters('mewz_wcas_limit_stock_status', $product->get_stock_status('edit') === 'instock', $product);
	}

	public function get_manage_stock($value, \WC_Product $product)
	{
		if ($value || !self::limit_product($product) || !self::limit_stock_status($product)) {
			return $value;
		}

		if (
			(!Products::is_product_excluded($product) && Limits::get_stock_limit($product))
			|| Products::get_multiplier($product, 'product') !== 1.00
		) {
			$value = true;

			// save original 'stock_status' before `$product->validate_props()` changes it in 'view' context (with filters) before saving!!
			Products::set_prop($product, 'original_stock_status', $product->get_stock_status('edit'));
		}

		return $value;
	}

	public function get_stock_quantity($value, \WC_Product $product)
	{
		if (!self::limit_product($product)) {
			return $value;
		}

		Products::incr_prop($product, 'bypass_limits');

		$manage_stock = $product->get_manage_stock();

		if ($manage_stock || self::limit_stock_status($product)) {
			if (apply_filters('mewz_wcas_limit_product_stock_quantity', true, $product, $value) && !Products::is_product_excluded($product)) {
				$limit = Limits::get_stock_limit($product);
			} else {
				$limit = false;
			}

			// get product multiplier if stock management is enabled on product (variation or parent / product-level)
			$multiplier = $manage_stock ? Products::get_multiplier($product, 'product') : 1.00;
			$has_multiplier = $multiplier !== 1.00;

			// apply the variation parent stock quantity filter because overriding `manage_stock` causes it to be skipped
			if (($limit || $has_multiplier) && $manage_stock === 'parent' && $product instanceof \WC_Product_Variation && $parent_data = $product->get_parent_data()) {
				$value = apply_filters('woocommerce_product_variation_get_stock_quantity', $parent_data['stock_quantity'], $product);
			}

			// apply product multiplier (can increase or decrease quantity)
			if ($has_multiplier) {
				$value = Matches::calc_limit_qty($value, $multiplier);
			}

			if ($limit && (!$manage_stock || $limit['limit_qty'] < $value)) {
				$value = $limit['limit_qty'];
			}
		}

		Products::decr_prop($product, 'bypass_limits');

		return $value;
	}

	public function get_stock_status($value, \WC_Product $product)
	{
		if (!$value || $value === 'outofstock' || !self::limit_product($product, true)) {
			return $value;
		}

		if ($override_value = Products::get_prop($product, 'stock_status')) {
			return $override_value;
		}

		Products::incr_prop($product, 'bypass_limits');
		$manage_stock = $product->get_manage_stock();
		$override_status = ($manage_stock || self::limit_stock_status($product)) && !$product->backorders_allowed();
		Products::decr_prop($product, 'bypass_limits');

		if (!$override_status) {
			return $value;
		}

		if ($product instanceof \WC_Product_Variable) {
			return $this->variable_get_stock_status($value, $product);
		}

		if ($manage_stock) {
			$quantity = Products::without_limits($product, 'get_stock_quantity');

			if (($multiplier = Products::get_multiplier($product, 'product')) !== 1.00) {
				$quantity = Matches::calc_limit_qty($quantity, $multiplier);
			}

			if ($quantity < 1 || $quantity <= WooCommerce::no_stock_amount()) {
				return 'outofstock';
			}
		}

		if (!Products::is_product_excluded($product)) {
			$limit = Limits::get_stock_limit($product);

			if ($limit && ($limit['limit_qty'] < 1 || $limit['limit_qty'] <= WooCommerce::no_stock_amount())) {
				return 'outofstock';
			}
		}

		return $value;
	}

	protected function variable_get_stock_status($value, \WC_Product $product)
	{
		// use synced product visibility for variable product stock status, if it's enabled and not bypassed
		if (!Products::get_prop($product, 'bypass_limits_product_visibility') && Settings::sync_product_visibility_bool(true)) {
			return is_object_in_term($product->get_id(), 'product_visibility', 'outofstock') ? 'outofstock' : $value;
		}

		$cache_key = 'variable_outofstock';
		$cache_tags = ['stock', 'match_rules', 'multipliers', 'product_' . $product->get_id()];

		$outofstock = $this->cache->get($cache_key, $cache_tags);

		if ($outofstock === null) {
			$outofstock = 1;

			foreach ($product->get_children() as $variation_id) {
				if (($variation = wc_get_product($variation_id)) && $variation->is_in_stock()) {
					$outofstock = 0;
					break;
				}
			}

			$this->cache->set($cache_key, $outofstock, $cache_tags);
		}

		return $outofstock ? 'outofstock' : $value;
	}

	public function variation_get_backorders($value, \WC_Product_Variation $variation)
	{
		if (!self::$enabled || Products::get_prop($variation, 'bypass_limits') > 0 || Products::is_product_excluded($variation)) {
			return $value;
		}

		Products::incr_prop($variation, 'bypass_limits');

		if ($variation->get_manage_stock() === 'parent' && $parent_data = $variation->get_parent_data()) {
			$value = apply_filters('woocommerce_product_variation_get_backorders', $parent_data['backorders'], $variation);
		}

		Products::decr_prop($variation, 'bypass_limits');

		return $value;
	}

	public function get_image_id($value, \WC_Product $product)
	{
		if ($value || !self::limit_product($product)) {
			return $value;
		}

		$limit = Limits::get_stock_limit($product);

		if (!empty($limit['image_id'])) {
			$value = $limit['image_id'];
		}

		return $value;
	}

	public function get_visible_children($variation_ids, $product, $visible_only = false)
	{
		if (!$visible_only || !self::limit_product($product, true)) {
			return $variation_ids;
		}

		$cache_key = 'visible_children';
		$cache_tags = ['stock', 'match_rules', 'multipliers', 'product_' . $product->get_id()];

		$visible_ids = $this->cache->get($cache_key, $cache_tags);

		if (!is_array($visible_ids)) {
			_prime_post_caches($variation_ids);
			$visible_ids = [];

			foreach ($variation_ids as $variation_id) {
				if (($variation = wc_get_product($variation_id)) && $variation->is_in_stock()) {
					$visible_ids[] = $variation_id;
				}
			}

			$this->cache->set($cache_key, $visible_ids, $cache_tags);
		}

		return $visible_ids;
	}

	public function order_item_product($product, $order_item)
	{
		/**
		 * Don't limit order item product stock else stock reduction calculations will be thrown out.
		 *
		 * But we still need to limit stock when the order item is being created, so that things like
		 * backorder meta can be added accordingly.
		 */
		if ($product && $order_item && $order_item->get_id()) {
			Products::incr_prop($product, 'bypass_limits');
		}

		return $product;
	}

	public function allow_backorders($value, \WC_Product $product)
	{
		if ($value !== 'no' || !self::$enabled || !Products::is_valid_product($product, true) || Products::is_product_excluded($product)) {
			return $value;
		}

		if (
			!$product->get_manage_stock('edit')
			&& self::limit_stock_status($product)
			&& (
				Products::get_prop($product, 'allow_backorders_override')
				|| Limits::get_stock_limit($product)
			)
		) {
			$value = Settings::allow_backorders(true);
		}

		return $value;
	}
}
